import * as React from "react";
import { SVG, SVGAsset } from "twitch-core-ui";
import "./styles.scss";
import { GameState, DynamoTypes } from "sumo-game-logic";
import { SumoGameView } from "./components/SumoGameView";
import { Matchmaking } from "extension_game/components/Matchmaking";
import * as WebFont from "webfontloader";

export interface Props {
    isPrime: boolean,
    opaqueUserId: string,
    displayedWinner: DynamoTypes.PlayerMetadata,
    ongoingSessionExists: boolean,
    gameState: GameState,
    newGameStateReceived: boolean,
    maxPlayersPerGame: number,
    matchmaking: number,
    errorMessage: string,
    closeGame: () => void,
    leaveGame: () => void,
}

export class ExtensionGameComponent extends React.Component<Props> {

    public render() {
        // Preload our game fonts before we need them
        WebFont.load({
            custom: {
                families: ['game-font', 'game-font-countdown']
            }
        });

        return (
            <div className="extension-game">
                <div className="extension-game__game-window">
                    {this.renderCloseButton()}
                    {this.renderGame()}
                </div>
            </div>
        );
    }

    private renderGame() {
        if (!!this.props.errorMessage) {
            return this.renderGameError();
        }
        else if (this.userInGameEndScreen()) {
            return this.renderEndGameScreen();
        }
        else if (this.userInGame()) {
            return this.renderUserInGame();
        }
        else if (this.ongoingSession()) {
            return this.renderStartupScreen();
        }
        else {
            return this.renderGameDisabled();
        }
    }

    private renderCloseButton() {
        return (
            <div className="extension-game__game-window__close-game"> 
                <button className="extension-game__game-window__close-game__button" onClick={this.props.closeGame}>
                    <SVG asset={SVGAsset.Close} />
                </button>
            </div>
        );
    }

    private renderGameError() {
        return (
            <div className="extension-game__game-window__text-based-menu">
                {this.props.errorMessage}
            </div>
        )
    }

    private renderUserInGame() {
        return (
            <SumoGameView/>
        );
    }

    private renderEndGameScreen() {
        const wonTheGame = this.props.opaqueUserId == this.props.displayedWinner.opaqueUserID;
        let endGameMessage = (this.props.displayedWinner.username == ""? "An anonymous user" : this.props.displayedWinner.username) + " wins!";
        if (wonTheGame) {
            endGameMessage = "You win!";
        }
        return (
            <div className="extension-game__game-window__text-based-menu">
                <div className={`extension-game__game-window__end-game__results 
                               ${wonTheGame? 'extension-game__game-window__end-game__winner' : 'extension-game__game-window__end-game__loser'}`}>
                    {endGameMessage}
                </div>
                <button className="extension-game__menu-button" onClick={this.props.leaveGame}>
                    Main menu
                </button>
            </div>
        );
    }

    private renderStartupScreen() {
        return (
            <div className="extension-game__game-window__main-menu">
                <div className="extension-game__game-window__main-menu__title">
                    <img 
                        className="extension-game__game-window__main-menu__title__title-art"
                        src="https://m.media-amazon.com/images/G/01/sm/reminder-extension/sumo/sumo-title-art._CB418300675_.png"
                        alt="Prime Sumo title art"
                    />
                </div>
                <div className="extension-game__game-window__main-menu__how-to-play">
                    <div className="extension-game__game-window__main-menu__how-to-play__text">
                        HOW TO PLAY
                    </div>
                    <img className="extension-game__game-window__main-menu__how-to-play__graphic" alt="how to play" src="https://m.media-amazon.com/images/G/01/sm/reminder-extension/sumo/sumo_tutorial_animation._CB1550532119_.gif"/>
                </div>
                <div className="extension-game__game-window__main-menu__matchmaking">
                    <Matchmaking />
                </div>
            </div>
        );
    }

    private renderGameDisabled() {
        return (
            <div className="extension-game__game-window__text-based-menu">
                The streamer has disabled Prime Gaming Sumo. Ask your streamer to enable it to start playing!
            </div>
        );
    }

    private userInGame() {
        return this.props.gameState != null && this.props.gameState.gameID != null;
    }

    private userInGameEndScreen() {
        return this.userInGame() && this.props.displayedWinner != null;
    }

    private ongoingSession() {
        return this.props.ongoingSessionExists;
    }
};