import * as React from "react";
import "./styles.scss";
import * as Spinner from "react-spinkit";

export interface Props {
    automaticMatchStartTimestamp: number,
    matchmaking: boolean,
    matchmakingPlayersCount: number,
    activePlayersCount: number,
    maxPlayersPerGame: number,
    getCurrentServerTime: () => number,
    startMatchmaking: () => void,
    cancelMatchmaking: () => void
}

export interface State {
    timeUntilMatchStart: number,
}

export class MatchmakingComponent extends React.Component<Props, State> {
    private countdownTimer;

    constructor(props) {
        super(props);

        this.state = {
            timeUntilMatchStart: null,
        }
    }

    public componentWillUnmount() {
        this.clearCountdownTimer();
    }

    public render() {
        if (this.gameStartIsCountingDown()) {
            this.startCountdownTimer();
            return this.renderGameStartCountingDown();
        }
        else {
            this.clearCountdownTimer();
            if (this.waitingForMorePlayers()) {
                return this.renderWaitingForMorePlayers();
            }
            else {
                return this.renderStartMatchmaking();
            }
        }
    }

    private renderGameStartCountingDown() {
        return (
            <div className="matchmaking">
                {this.renderCountingDownStatus()}
                {this.renderCancelMatchmakingButton()}
            </div>
        );
    }


    private renderWaitingForMorePlayers() {
        return (
            <div className="matchmaking">
                {this.renderWaitingForMorePlayersStatus()}
                {this.renderCancelMatchmakingButton()}
            </div>
        );
    }

    private renderStartMatchmaking() {
        return (
            <div className="matchmaking">
                {this.renderPlayerCountStatus()}
                {this.renderStartMatchmakingButton()}
            </div>
        );
    }

    private renderPlayerCountStatus() {
        return (
            <span className="matchmaking__no-wrap-text">
                {this.props.activePlayersCount} playing on this channel now
            </span>
        );
    }

    private renderCountingDownStatus() {
        return (
            <div className="matchmaking__status">
                {this.renderMatchmakingStatusSpinner()}
                {this.renderCountingDownStatusInfo()}
            </div>
        );
    }

    private renderCountingDownStatusInfo() {
        return (
            <div className="matchmaking__status__info">
                <span className="matchmaking__no-wrap-text">Starting match in {this.state.timeUntilMatchStart}</span>
                {this.renderPlayersQueued()}
            </div>
        );
    }

    private renderWaitingForMorePlayersStatus() {
        return (
            <div className="matchmaking__status">
                {this.renderMatchmakingStatusSpinner()}
                {this.renderWaitingForMorePlayersStatusInfo()}
            </div>
        );
    }

    private renderWaitingForMorePlayersStatusInfo() {
        return (
            <div className="matchmaking__status__info">
                MATCHMAKING...
                {this.renderPlayersQueued()}
            </div>
        );
    }

    private renderPlayersQueued() {
        return (
            <span className="matchmaking__no-wrap-text">{this.props.matchmakingPlayersCount} {this.props.matchmakingPlayersCount == 1 ? "player" : "players"} queued up!</span>
        );
    }

    private renderStartMatchmakingButton() {
        return (
            <button className="extension-game__menu-button matchmaking__button" onClick={this.props.startMatchmaking}>
                I'M READY!
            </button>
        );
    }

    private renderCancelMatchmakingButton() {
        return (
            <button className="extension-game__menu-button matchmaking__button" onClick={this.props.cancelMatchmaking}>
                CANCEL
            </button>
        );
    }

    private renderMatchmakingStatusSpinner() {
        return (
            <Spinner 
                className={'matchmaking__status__spinner'}
                name='cube-grid'
                fadeIn={'none'}
                color={'#6441A4'}
            />
        );
    }

    private startCountdownTimer() {
        if (this.countdownTimer == null) {
            setTimeout(this.updateTimer, 0); // do not update immediately in render()
            this.countdownTimer = setInterval(this.updateTimer, 1000);
        }
    }

    private clearCountdownTimer() {
        clearInterval(this.countdownTimer);
        this.countdownTimer = null;
    }

    private updateTimer = () => {
        const secondsToStart = (this.props.automaticMatchStartTimestamp - this.props.getCurrentServerTime())/1000.0;
        this.setState({
            timeUntilMatchStart: Math.max(0, Math.ceil(secondsToStart)),
        });
    }

    private gameStartIsCountingDown() {
        return this.props.matchmaking && this.props.automaticMatchStartTimestamp != null;
    }

    private waitingForMorePlayers() {
        return this.props.matchmaking && this.props.automaticMatchStartTimestamp == null;
    }
};
