import * as PIXI from "pixi.js";
import { GameState } from "sumo-game-logic";
import { DARK_TWITCH_PURPLE_HEX, LIGHT_TWITCH_PURPLE_HEX, MY_COLOR_HEX } from "../Helpers/colorHandler";
import { STATUS_TEXT_STYLE } from "../Helpers/textStyleHandler";

export enum StatusText {
    DRUMROLL = "Gathering inputs",
    MAKE_INPUT = "Click and drag",
    DEAD = "You died"
}

export class GameUIView {

    private static readonly PROGRESS_BAR_HEIGHT = 12;
    private static readonly PROGRESS_BAR_ALPHA = 0.7;

    public gameState: GameState.GameState;
    private parentContainer: PIXI.Container;

    public UIWidth: number;
    public UIHeight: number;

    // Progress bar
    private inputPhaseProgressBarGraphic: PIXI.Graphics;

    // Drumroll indicator timer text
    private static readonly DRUMROLL_TEXT_DOT_CHANGE_INTERVAL_MS = 400; // animation speed for the "..." in the drumroll text
    private static readonly MAKE_INPUT_TEXT_ALPHA_ANIMATION_PERIOD_MS = 1500; // animation speed for the alpha tween for the make input text
    private static readonly DRUMROLL_TEXT_COLOR = LIGHT_TWITCH_PURPLE_HEX;
    private static readonly MAKE_INPUT_TEXT_COLOR = MY_COLOR_HEX;
    private static readonly DEAD_TEXT_COLOR = LIGHT_TWITCH_PURPLE_HEX;
    private static readonly STATUS_TEXT_ALPHA = 0.7;
    private static readonly STATUS_TEXT_HEIGHT = 30.0;
    private makeInputText: PIXI.Text;
    private drumrollText: PIXI.Text;
    private deadText: PIXI.Text;

    constructor(gameState: GameState.GameState, parentContainer: PIXI.Container, UIWidth: number, UIHeight: number) {
        this.gameState = gameState;
        this.parentContainer = parentContainer;
        this.UIWidth = UIWidth;
        this.UIHeight = UIHeight;

        // create progress bar sprite
        this.inputPhaseProgressBarGraphic = new PIXI.Graphics();
        this.inputPhaseProgressBarGraphic.alpha = GameUIView.PROGRESS_BAR_ALPHA;
        this.inputPhaseProgressBarGraphic.renderable = false;
        this.updateProgressBarGraphic(0);
        this.parentContainer.addChild(this.inputPhaseProgressBarGraphic);

        // create status text
        this.makeInputText = this.createStatusText(StatusText.MAKE_INPUT, GameUIView.MAKE_INPUT_TEXT_COLOR);
        this.drumrollText = this.createStatusText(StatusText.DRUMROLL, GameUIView.DRUMROLL_TEXT_COLOR);
        this.deadText = this.createStatusText(StatusText.DEAD, GameUIView.DEAD_TEXT_COLOR);
    }

    private createStatusText = (message, color) => {
        const text = new PIXI.Text(message, STATUS_TEXT_STYLE);
        text.alpha = GameUIView.STATUS_TEXT_ALPHA;
        const textScale = GameUIView.STATUS_TEXT_HEIGHT / text.height;
        text.scale.set(textScale, textScale);
        text.anchor.set(0, 0.5);
        this.centerStatusText(text);
        text.renderable = false;
        text.tint = color;
        this.parentContainer.addChild(text);
        return text;
    }

    private centerStatusText(text: PIXI.Text) {
        text.position.set(this.UIWidth/2.0 - text.width/2.0, this.UIHeight - GameUIView.STATUS_TEXT_HEIGHT * 1.3);
    }

    public updateDimensions(newWidth, newHeight) {
        this.UIWidth = newWidth;
        this.UIHeight = newHeight;
        this.centerStatusText(this.drumrollText);
        this.centerStatusText(this.makeInputText);
        this.centerStatusText(this.deadText);
    }

    public updateProgressBarGraphic(currentServerTime: number) {
        const inputPhaseStartTimestamp = this.gameState.getNextRoundInputStartTimestamp();
        const inputPhaseEndTimestamp = this.gameState.getNextRoundInputCutoffTimestamp();

        let progress = (currentServerTime - inputPhaseStartTimestamp) / (inputPhaseEndTimestamp - inputPhaseStartTimestamp);
        progress = Math.max(0, Math.min(1, progress));
        this.inputPhaseProgressBarGraphic.clear();

        // Progress bar container
        this.inputPhaseProgressBarGraphic.beginFill(DARK_TWITCH_PURPLE_HEX);
        this.inputPhaseProgressBarGraphic.drawRect(0, 0, this.UIWidth, GameUIView.PROGRESS_BAR_HEIGHT);
        this.inputPhaseProgressBarGraphic.endFill();

        // Progress bar fill
        this.inputPhaseProgressBarGraphic.beginFill(LIGHT_TWITCH_PURPLE_HEX);
        this.inputPhaseProgressBarGraphic.drawRect(0, 0, progress * this.UIWidth, GameUIView.PROGRESS_BAR_HEIGHT);
        this.inputPhaseProgressBarGraphic.endFill();
    }

    private updateDisplayedStatusText(gamePhase: GameState.GamePhase, dead: boolean) {
        if (dead) {
            this.drumrollText.renderable = false;
            this.makeInputText.renderable = false;
            this.deadText.renderable = true;
            return;
        }

        this.deadText.renderable = false;
        switch (gamePhase) {
            case GameState.GamePhase.INPUT_PHASE:
                this.drumrollText.renderable = false;
                this.makeInputText.renderable = true;
                this.updateMakeInputText();
                break;
            case GameState.GamePhase.DRUMROLL_PHASE:
                this.drumrollText.renderable = true;
                this.makeInputText.renderable = false;
                this.updateDrumrollText();
                break;
            case GameState.GamePhase.RESULTS_PHASE:
                this.drumrollText.renderable = false;
                this.makeInputText.renderable = false;
                break;
        }
    }

    public update(gamePhase: GameState.GamePhase, currentServerTime: number, dead: boolean) {

        this.updateDisplayedStatusText(gamePhase, dead);

        switch (gamePhase) {
            case GameState.GamePhase.INPUT_PHASE:
                this.inputPhaseProgressBarGraphic.renderable = true;
                this.updateProgressBarGraphic(currentServerTime);
                break;
            case GameState.GamePhase.DRUMROLL_PHASE:
                this.inputPhaseProgressBarGraphic.renderable = false;
                break;
            case GameState.GamePhase.RESULTS_PHASE:
                this.inputPhaseProgressBarGraphic.renderable = false;
                break;
        }
    }

    public updateDrumrollText() {
        // animate three dots
        const numDots = (Date.now() / GameUIView.DRUMROLL_TEXT_DOT_CHANGE_INTERVAL_MS) % 4;
        const dotString = '.'.repeat(numDots);
        this.drumrollText.text = StatusText.DRUMROLL + dotString;
    }

    public updateMakeInputText() {
        // animate alpha to go between 0 and GameUIView.STATUS_TEXT_ALPHA
        const alphaLerp = (Math.sin((Date.now() / GameUIView.MAKE_INPUT_TEXT_ALPHA_ANIMATION_PERIOD_MS) * 2 * Math.PI) + 1) / 2.0;
        const alpha = GameUIView.STATUS_TEXT_ALPHA * alphaLerp;
        this.makeInputText.alpha = alpha;
    }
}