import * as PIXI from "pixi.js";
import { GameState } from "sumo-game-logic";
import { VERY_DARK_TWITCH_PURPLE_HEX } from "../Helpers/colorHandler";
import { COUNTDOWN_TEXT_STYLE } from "../Helpers/textStyleHandler";

export class MapView {

    public gameState: GameState.GameState;
    private mapGraphic: PIXI.Graphics;
    private parentContainer: PIXI.Container;

    // Input phase timer text
    private static readonly COUNTDOWN_TEXT_ALPHA = 0.2;
    private countdownText: PIXI.Text;

    constructor(gameState: GameState.GameState, parentContainer: PIXI.Container, currentServerTime: number) {
        this.gameState = gameState;
        this.parentContainer = parentContainer;

        // create map graphic
        this.mapGraphic = new PIXI.Graphics();
        this.parentContainer.addChild(this.mapGraphic);

        // create countdown text
        this.countdownText = new PIXI.Text('', COUNTDOWN_TEXT_STYLE);
        this.countdownText.alpha = MapView.COUNTDOWN_TEXT_ALPHA;
        this.countdownText.anchor.set(0.5, 0.5);
        this.countdownText.renderable = false;
        this.parentContainer.addChild(this.countdownText);

        this.update(currentServerTime);
    }

    public update(currentServerTime) {
        this.updateMapGraphic();
        this.updateCountdownTimer(currentServerTime);
    }

    public updateMapGraphic() {
        this.mapGraphic.clear();

        this.mapGraphic.beginFill(VERY_DARK_TWITCH_PURPLE_HEX);

        const radius = this.gameState.gameMap.getSafeZoneRadius();
        this.mapGraphic.drawRect(-radius, -radius, radius*2.0, radius*2.0);

        this.mapGraphic.endFill();
    }

    public updateCountdownTimer(currentServerTime) {
        const msUntilInputEnd = this.gameState.getNextRoundInputCutoffTimestamp() - currentServerTime;
        const numberToDisplay = Math.ceil(msUntilInputEnd / 1000.0);
        const countdownTextContents = msUntilInputEnd < 0? '' : '' + numberToDisplay;
        this.setCountdownText(countdownTextContents);
        
        // Resize text to fit inside map
        const targetHeight = this.gameState.gameMap.getSafeZoneRadius() * 2.0;
        const currentHeight = this.countdownText.height;
        const newScale = this.countdownText.scale.x * (targetHeight*1.0) / currentHeight;
        this.countdownText.scale.set(newScale, newScale);
    }

    public hideCountdownTimer() {
        this.countdownText.renderable = false;
    }

    public showCountdownTimer() {
        this.countdownText.renderable = true;
    }
    
    private setCountdownText(contents: string) {
        this.countdownText.text = contents;
    }
}