import * as PIXI from "pixi.js";
import { Player } from "sumo-game-logic";
import { getPlayerColor, getTrailTexture } from "../Helpers/colorHandler";
import { getPlayerNameTextStyle } from "../Helpers/textStyleHandler";

export class PlayerView {

    public playerState: Player.Player;
    private deadOnRoundStart: boolean;

    // Player object graphic vars
    public playerObjectGraphic: PIXI.Graphics;
    private color;

    // Trail vars
    private static readonly NUM_TRAIL_POINTS: number = 30;
    private trailPoints: PIXI.Point[];
    private trailRope;

    // UI vars
    private static readonly UI_ALPHA = 0.85;
    // Input arrow
    public inputArrowGraphic: PIXI.Graphics;
    private static readonly ARROW_BODY_WIDTH = 6.5;
    private static readonly ARROW_HEAD_WIDTH = PlayerView.ARROW_BODY_WIDTH * 3.1;
    private static readonly ARROW_HEAD_LENGTH = PlayerView.ARROW_BODY_WIDTH * 2.25;
    // Player name text
    private static readonly PLAYER_NAME_HEIGHT_ABOVE_PLAYER_OBJECT = - (Player.Player.PLAYER_RADIUS + 25);
    private playerNameText: PIXI.Text;
    // Death graphic
    private static readonly PLAYER_DEATH_MARK_IMAGE_PATH = 'https://m.media-amazon.com/images/G/01/sm/reminder-extension/sumo/sumo_death_mark._CB1550524879_.png';
    private static readonly PLAYER_DEATH_MARK_STARTING_SIZE = Player.Player.PLAYER_RADIUS * 7; // death mark appears at this size
    private static readonly PLAYER_DEATH_MARK_TARGET_SIZE = 0; // death mark ends on this size
    private static readonly PLAYER_DEATH_MARK_SIZE_LERP_FACTOR = 0.3; // mark changes size at this speed every frame (1 is instant, 0 is no size change)
    private deathMarkSprite: PIXI.Sprite;

    private parentContainer: PIXI.Container;
    private playerContainer: PIXI.Container;

    constructor(playerState: Player.Player, isMyPlayer: boolean, parentContainer: PIXI.Container) {
        this.color = getPlayerColor(isMyPlayer);

        this.parentContainer = parentContainer;
        this.playerContainer = new PIXI.Container();

        // create input graphic
        this.inputArrowGraphic = new PIXI.Graphics();
        this.inputArrowGraphic.alpha = PlayerView.UI_ALPHA;
        this.inputArrowGraphic.renderable = false;

        // create player graphic
        this.playerObjectGraphic = new PIXI.Graphics();
        this.createPlayerGraphic();

        // create trail graphic
        this.trailPoints = [];
        for (let i = 0; i < PlayerView.NUM_TRAIL_POINTS; i++) {
            this.trailPoints.push(new PIXI.Point(0, 0));
        }
        const texture = getTrailTexture(isMyPlayer);
        this.trailRope = new PIXI.mesh.Rope(texture, this.trailPoints);
        this.trailRope.blendmode = PIXI.BLEND_MODES.ADD;
        this.trailRope.renderable = false;

        // create player name text
        let displayedName = playerState.getPlayerMetadata().username;
        if (isMyPlayer) {
            displayedName = "You";
        }
        this.playerNameText = new PIXI.Text(displayedName, getPlayerNameTextStyle(isMyPlayer));
        this.playerNameText.anchor.set(0.5, 0);
        this.playerNameText.alpha = PlayerView.UI_ALPHA;
        this.playerNameText.position.set(0, PlayerView.PLAYER_NAME_HEIGHT_ABOVE_PLAYER_OBJECT);
        this.playerContainer.addChild(this.playerNameText);

        // create death graphic
        this.deathMarkSprite = PIXI.Sprite.fromImage(PlayerView.PLAYER_DEATH_MARK_IMAGE_PATH);
        this.deathMarkSprite.anchor.set(0.5, 0.5);
        this.deathMarkSprite.alpha = PlayerView.UI_ALPHA;
        this.deathMarkSprite.tint = this.color;
        this.deathMarkSprite.width = PlayerView.PLAYER_DEATH_MARK_STARTING_SIZE;
        this.playerContainer.addChild(this.deathMarkSprite);

        // add to container in correct z-order
        this.parentContainer.addChild(this.trailRope);
        this.parentContainer.addChild(this.playerContainer);
        this.playerContainer.addChild(this.inputArrowGraphic);
        this.playerContainer.addChild(this.playerObjectGraphic);

        this.handleNewPlayerState(playerState);
    }

    public handleNewPlayerState(playerState: Player.Player) {
        this.playerState = playerState;
        
        this.deadOnRoundStart = this.playerState.dead;
        if (this.deadOnRoundStart) {
            this.removeFromScene();
        } else {
            // input
            this.updateInputGraphic(this.playerState.roundInput);
    
            // death sprite
            this.deathMarkSprite.renderable = this.playerState.dead;
        }
    }

    public createPlayerGraphic() {
        this.playerObjectGraphic.clear();
        this.playerObjectGraphic.beginFill(this.color);

        this.playerObjectGraphic.drawCircle(0, 0, Player.Player.PLAYER_RADIUS);

        this.playerObjectGraphic.endFill();
    }

    public updateInputGraphic(input = {x: 0, y: 0}) {
        this.inputArrowGraphic.clear();

        if (input.x == 0 && input.y == 0) {
            return;
        }

        this.inputArrowGraphic.beginFill(this.color);

        const fromX = 0;
        const fromY = 0;
        const toX = input.x;
        const toY = input.y;

        // 1. Draw the arrow horizontally
        // arrow body
        const bodyLength = Math.sqrt(toX*toX + toY*toY) - PlayerView.ARROW_HEAD_LENGTH;
        this.inputArrowGraphic.drawRect(fromX, fromY - PlayerView.ARROW_BODY_WIDTH/2.0, bodyLength, PlayerView.ARROW_BODY_WIDTH);

        // arrow head
        const headPointX = fromX + bodyLength + PlayerView.ARROW_HEAD_LENGTH;
        const headPointY = fromY;
        this.inputArrowGraphic.moveTo(headPointX, headPointY);
        this.inputArrowGraphic.lineTo(headPointX - PlayerView.ARROW_HEAD_LENGTH, headPointY - PlayerView.ARROW_HEAD_WIDTH/2.0);
        this.inputArrowGraphic.lineTo(headPointX - PlayerView.ARROW_HEAD_LENGTH, headPointY + PlayerView.ARROW_HEAD_WIDTH/2.0);
        this.inputArrowGraphic.lineTo(headPointX, headPointY);

        this.inputArrowGraphic.endFill();

        // 2. Rotate the arrow to face the ending position
        this.inputArrowGraphic.rotation = Math.atan2(toY-fromY, toX-fromX);
    }

    private updateTrail() {
        // Set current player position as trail head (recycling our point objects)
        const newPoint = this.trailPoints.pop();
        newPoint.set(this.playerContainer.position.x, this.playerContainer.position.y);
        this.trailPoints.unshift(newPoint);
    }

    public update() {
        if (this.deadOnRoundStart) {
            return;
        }

        if (this.playerState.dead) {
            this.playerObjectGraphic.renderable = false;
            this.trailRope.renderable = false;
            this.playerNameText.renderable = false;

            // animate death mark
            this.deathMarkSprite.renderable = true;
            const distanceToTargetSize = PlayerView.PLAYER_DEATH_MARK_TARGET_SIZE - this.deathMarkSprite.width;
            const newMarkSize = this.deathMarkSprite.width + distanceToTargetSize * PlayerView.PLAYER_DEATH_MARK_SIZE_LERP_FACTOR;
            this.deathMarkSprite.width = this.deathMarkSprite.height = newMarkSize;
        }

        this.updateTrail();
        this.playerContainer.position.set(
            this.playerState.currentPosition.x, 
            this.playerState.currentPosition.y
        );
    }

    public showTrail() {
        if (this.trailRope != null) {
            this.trailRope.renderable = true;
        }
    }
    public hideTrail() {
        if (this.trailRope != null) {
            this.trailRope.renderable = false;
        }
    }
    public resetTrail() {
        this.trailPoints.forEach((point) => {
            point.set(this.playerState.currentPosition);
        })
    }

    public hideInputArrowGraphic() {
        if (this.inputArrowGraphic != null) {
            this.inputArrowGraphic.renderable = false;
        }
    }
    public showInputArrowGraphic() {
        if (this.inputArrowGraphic != null) {
            this.inputArrowGraphic.renderable = true;
        }
    }
    public resetInputArrowGraphic() {
        this.inputArrowGraphic.clear();
    }

    public removeFromScene() {
        if (this.playerContainer != null) {
            this.parentContainer.removeChild(this.playerContainer);
        }
        if (this.trailRope != null) {
            this.parentContainer.removeChild(this.trailRope);
        }
    }
}