import * as PIXI from "pixi.js";
import { Player } from "sumo-game-logic";

// Map color
export const TWITCH_PURPLE_HEX = 0x6441a4;
export const LIGHT_TWITCH_PURPLE_HEX = 0xb9a3e3;
export const DARK_TWITCH_PURPLE_HEX = 0x3b2064;
export const VERY_DARK_TWITCH_PURPLE_HEX = 0x251a36;

// Player colors and textures
const MY_COLOR_RGB = [241, 196, 15]; // Yellow
const ENEMY_COLOR_RGB = [121, 80, 186]; // Slightly lighter twitch purple
const TRAIL_STARTING_ALPHA = 0.7;
const TRAIL_ENDING_ALPHA = 0;

/** Converts an RGB array from using 255 as max value to using 1 as max value */
function RGB255ToRGB1(color: number[]) {
    return ([
        color[0]/255.0, 
        color[1]/255.0, 
        color[2]/255.0
    ]);
}

/** Creates a transparency gradient texture from a given color. */
function createTransparencyGradientTexture(color: number[]) {
    // Create new canvas (required for every new texture)
    const canvas = document.createElement('canvas');
    canvas.width  = 200;
    canvas.height = Player.Player.PLAYER_RADIUS * 2;
    const ctx = canvas.getContext('2d');

    // Create and apply gradient to canvas
    const gradient = ctx.createLinearGradient(0, 0, canvas.width, 0);
    const colorString = `${color[0]}, ${color[1]}, ${color[2]}`;
    gradient.addColorStop(0, `rgba(${colorString}, ${TRAIL_STARTING_ALPHA})`);
    gradient.addColorStop(1, `rgba(${colorString}, ${TRAIL_ENDING_ALPHA})`);
    ctx.fillStyle = gradient;
    ctx.fillRect(0, 0, canvas.width, canvas.height);

    // Create and return texture
    return PIXI.Texture.fromCanvas(canvas);
}

const MY_TRAIL_TEXTURE = createTransparencyGradientTexture(MY_COLOR_RGB);
const ENEMY_TRAIL_TEXTURE = createTransparencyGradientTexture(ENEMY_COLOR_RGB);

export const MY_COLOR_HEX = PIXI.utils.rgb2hex(RGB255ToRGB1(MY_COLOR_RGB));
export const ENEMY_COLOR_HEX = PIXI.utils.rgb2hex(RGB255ToRGB1(ENEMY_COLOR_RGB));

export function getPlayerColor(isMyPlayer: boolean) {
    return isMyPlayer? MY_COLOR_HEX : ENEMY_COLOR_HEX;
}
export function getTrailTexture(isMyPlayer: boolean) {
    return isMyPlayer? MY_TRAIL_TEXTURE : ENEMY_TRAIL_TEXTURE;
}