import * as React from "react";
import "./styles.scss";
import { GameState, GameLogic, RoundResultsTimer, GameMap, Player } from "sumo-game-logic";
import { PlayerView } from "./GameObjectViews/playerView";
import { GameUIView } from "./GameObjectViews/gameUIView";
import { MapView } from "./GameObjectViews/mapView";
import * as PIXI from "pixi.js";

PIXI.utils.skipHello();

export interface Props {
    gameState: any,
    myOpaqueUserID: string,
    getCurrentServerTime: () => number,
    onGameInput: (input) => void,
    getGameSession: () => void,
    setDisplayedWinner: (displayedWinner: string) => void,
}

export class SumoGameViewComponent extends React.Component<Props> {

    // display
    private static readonly NON_SCALED_GAME_HEIGHT: number = GameMap.GameMap.DEFAULT_SAFE_ZONE_RADIUS * 2.5;
    private app: PIXI.Application;
    private gameCanvas: HTMLDivElement; // ref object to display the PIXI view in a react component
    private gameCamera: PIXI.Container;

    // current round input
    private inputMouseStartPos;
    private input;

    // state
    private gameStateModel: GameState.GameState;

    // views
    private playerViews: PlayerView[] = [];
    private myPlayerView: PlayerView;
    private gameUIView: GameUIView;
    private mapView: MapView;

    // Timeout id to ask for new state when the time is right
    private newStatePollTimeoutID;
  
    constructor(props) {
        super(props);

        this.inputMouseStartPos = null;
        this.input = {
            x: 0,
            y: 0
        };
    }

    private onInputStart = (event) => {
        if (this.isInputAllowed()) {
            this.inputMouseStartPos = { ...event.data.global };
            this.input.x = this.input.y = 0;
            this.myPlayerView.updateInputGraphic(this.input);
            this.myPlayerView.showInputArrowGraphic();
        }
    }

    private onInputEnd = () => {
        if (this.currentlyMakingInput()) {
            const inputToSend = {
              input: this.input, 
              roundNumber: this.gameStateModel.getRoundNumber() + 1
            };
            this.props.onGameInput(inputToSend);
            this.inputMouseStartPos = null;
        }
    }

    private onInputMove = (event) => {
        if (this.currentlyMakingInput()) {
            const currentInputMousePos = event.data.global;
            this.input.x = (currentInputMousePos.x - this.inputMouseStartPos.x) / this.app.stage.scale.x;
            this.input.y = (currentInputMousePos.y - this.inputMouseStartPos.y) / this.app.stage.scale.y;
            // cap out max input magnitude
            const inputMagnitudeSq = Math.pow(this.input.x, 2) + Math.pow(this.input.y, 2);
            if (inputMagnitudeSq > Math.pow(Player.Player.MAX_INPUT_MAGNITUDE, 2)) {
                const inputMagnitude = Math.sqrt(inputMagnitudeSq);
                const scaleFactor = Player.Player.MAX_INPUT_MAGNITUDE / inputMagnitude;
                this.input.x *= scaleFactor;
                this.input.y *= scaleFactor;
            }
            this.myPlayerView.updateInputGraphic(this.input);
        }
    }

    private serverAcceptsInputs = () => {
        return this.props.getCurrentServerTime() < this.gameStateModel.getNextRoundInputCutoffTimestamp();
    }

    private isInputAllowed = () => {
        return this.gameStateModel.getGamePhase() == GameState.GamePhase.INPUT_PHASE && this.serverAcceptsInputs() && !this.myPlayerView.playerState.dead;
    }

    private currentlyMakingInput = () => {
        return this.inputMouseStartPos != null;
    }

    public componentWillMount() {
        this.app = new PIXI.Application({
            transparent: true
        });
        this.app.stage = new PIXI.Container();

        // Add listeners for input
        this.app.renderer.plugins.interaction
            .on('pointerdown', this.onInputStart)
            .on('pointerup', this.onInputEnd)
            .on('pointerupoutside', this.onInputEnd)
            .on('pointermove', this.onInputMove);

        // setup game camera (to center the game on the viewer's screen)
        this.gameCamera = new PIXI.Container();
        this.app.stage.addChild(this.gameCamera);
    }

    // Resize function window
    private resize = () => {
        if (this.gameCanvas != null) {
            // resize the PIXI application to the div size
            const newWidth = this.gameCanvas.offsetWidth;
            const newHeight = this.gameCanvas.offsetHeight;
            this.app.renderer.resize(newWidth, newHeight);

            // scale the game displayed in the PIXI application
            const nonScaledGameHeight = SumoGameViewComponent.NON_SCALED_GAME_HEIGHT;
            const scaledGameHeight = this.gameCanvas.offsetHeight;
            const gameScale = scaledGameHeight / nonScaledGameHeight;
            this.app.stage.scale.set(gameScale, gameScale);

            // re-center all game objects
            this.gameCamera.position.set(nonScaledGameHeight/2.0, nonScaledGameHeight/2.0);

            if (this.gameUIView != null) {
                this.gameUIView.updateDimensions(newWidth / gameScale, newHeight / gameScale);
            }
        }
    }

    public componentDidMount() {
        this.gameCanvas.appendChild(this.app.view);
        this.app.start();
        this.app.ticker.add(delta => this.gameLoop(delta));

        window.addEventListener('resize', this.resize);
        this.resize();
    }

    public componentWillUnmount() {
        // stops the PIXI app render and update loop when the game isn't showing
        this.app.stop();
    }

    private gameLoop(delta) {
        this.gameUIView.update(this.gameStateModel.getGamePhase(), this.props.getCurrentServerTime(), this.myPlayerView.playerState.dead);
        switch (this.gameStateModel.getGamePhase()) {
            case GameState.GamePhase.INPUT_PHASE:
                this.onInputPhaseUpdate();
                break;
            case GameState.GamePhase.RESULTS_PHASE:
                this.onResultsPhaseUpdate();
                break;
        }
    }

    public onInputPhaseStart = () => {
        this.gameStateModel.setGamePhase(GameState.GamePhase.INPUT_PHASE);
        this.gameStateModel.reconcileWithExpectedEndState();
        this.updateGameObjectViews();
        this.mapView.showCountdownTimer();

        // hide player results UI
        this.playerViews.forEach((playerView) => {
            playerView.resetInputArrowGraphic();
            playerView.hideInputArrowGraphic();
            playerView.hideTrail();
        });
    }

    public onInputPhaseUpdate = () => {
        this.gameUIView.updateProgressBarGraphic(this.props.getCurrentServerTime());
        this.mapView.updateCountdownTimer(this.props.getCurrentServerTime());
        // Transition to drumroll phase if appropriate.
        if (!this.serverAcceptsInputs()) {
            this.onInputEnd(); // allow last second submission if player was dragging when input session ended
            this.onDrumrollPhaseStart();
        }
    }

    public onDrumrollPhaseStart = () => {
        this.gameStateModel.setGamePhase(GameState.GamePhase.DRUMROLL_PHASE);
        this.mapView.hideCountdownTimer();
    }

    public onResultsPhaseStart = () => {
        this.gameStateModel.setGamePhase(GameState.GamePhase.RESULTS_PHASE);
        GameLogic.GameLogic.startPhysicsSimulation(this.gameStateModel, false);
        this.onResultsPhaseUpdate();

        // show player results UI
        this.playerViews.forEach((playerView) => {
            playerView.showInputArrowGraphic();
            playerView.showTrail();
        });
    }

    public onResultsPhaseUpdate = () => {
        GameLogic.GameLogic.updateUntilGivenTime(this.gameStateModel, this.props.getCurrentServerTime());

        this.updateGameObjectViews();

        // Hide player inputs and drumroll text if we aren't in the SHOW_INPUTS phase of the results
        if (this.gameStateModel.roundResultsTimer.getPhaseForTime(this.props.getCurrentServerTime()) != RoundResultsTimer.ResultsPhase.SHOW_INPUTS) {
            this.playerViews.forEach((playerView) => {
                playerView.hideInputArrowGraphic();
            });
        } else {
            this.gameUIView.updateDrumrollText();
        }
    }

    private updateGameObjectViews = () => {
        this.mapView.update(this.props.getCurrentServerTime());
        this.playerViews.forEach((playerView) => {
            playerView.update();
        });
    }

    /** Transition to the appropriate game phase when the current round finishes. */
    public onRoundEnd = () => {
        const winnerMetadata = this.gameStateModel.getGameWinnerMetadata();
        if (winnerMetadata != null) {
            this.props.setDisplayedWinner(winnerMetadata);
        } else {
            this.onInputPhaseStart();
        }
    }

    public render() {
        if (this.props.gameState != null && this.props.gameState.storedState != null) {
            // Only make changes upon receiving a game state for a new round.
            // TODO: if is not new state, setup polling for new state after one second.
            if (this.gameStateModel == null || this.props.gameState.storedState.roundNumber > this.gameStateModel.getRoundNumber()) {
                this.gameStateModel = new GameState.GameState(this.props.gameState.storedState, true, this.onRoundEnd);

                const onFirstRound = this.gameStateModel.getRoundNumber() == 0;

                // Set a timeout to demand the next round state if the game isn't over yet.
                clearTimeout(this.newStatePollTimeoutID);
                if (this.gameStateModel.getGameWinnerMetadata() == null) {
                    const pollDelay = this.gameStateModel.getNextRoundEarliestStatePollTimestamp() - this.props.getCurrentServerTime();
                    this.newStatePollTimeoutID = setTimeout(this.props.getGameSession, pollDelay);
                }

                this.updateViewsFromGameState();

                // Transition to appropriate phase depending on round number.
                if (onFirstRound) {
                    this.onInputPhaseStart();
                } else {
                    this.onResultsPhaseStart();
                }
            }
        }
        
        // Display PIXI app in our React app
        let component = this;
        return (
            <div className="game" ref={(thisDiv) => {component.gameCanvas = thisDiv}} />
        );
    }

    private updateViewsFromGameState = () => {
        if (this.gameUIView == null) {
            this.gameUIView = new GameUIView(this.gameStateModel, this.app.stage, this.app.screen.width, this.app.screen.height);
        } else {
            this.gameUIView.gameState = this.gameStateModel;
        }

        if (this.mapView == null) {
            this.mapView = new MapView(this.gameStateModel, this.gameCamera, this.props.getCurrentServerTime());
        } else {
            this.mapView.gameState = this.gameStateModel;
        }
        
        this.gameStateModel.players.forEach((newPlayerState) => {
            const isMyPlayer = newPlayerState.getPlayerMetadata().opaqueUserID == this.props.myOpaqueUserID;

            let playerViewForModel = this.playerViews.find((view) => {
                return view.playerState.getPlayerMetadata().opaqueUserID == newPlayerState.getPlayerMetadata().opaqueUserID;
            });

            if (playerViewForModel != undefined) {
                // reuse existing player view if possible
                playerViewForModel.handleNewPlayerState(newPlayerState);
            } else {
                playerViewForModel = new PlayerView(newPlayerState, isMyPlayer, this.gameCamera);
                this.playerViews.push(playerViewForModel);
            }

            playerViewForModel.resetTrail();

            if (isMyPlayer) {
                this.myPlayerView = playerViewForModel;
            }
        });
    }
};