import { shallow } from "enzyme";
import { Location } from "core/types/CommunityService";
import { COMPONENT, OVERLAY } from "core/store/modules/Display";
import * as React from "react";
import { RootComponent, TestSelectors } from "./component";
import { StreamLocationPreview } from "live_config/components/StreamLocationPreview";
import { Dropdown } from "live_config/components/Dropdown";
import { CoreText } from "twitch-core-ui";
import { ErrorMessage } from "live_config/components/ErrorMessage";


interface PropOptions {
  errorMessage?: string;
  extensionLocation?: Location;
  theme?: string;
  typeOfExtension?: string;
  ongoingSessionExists?: boolean;
  channelIsWhitelistedForSumo?: boolean;
  onStartGameSession?: () => void;
  onEndGameSession?: () => void;
}

const getDefaultProps = (options?: PropOptions) => {
  return (
    <RootComponent
      errorMessage={ options && options.errorMessage }
      extensionLocation={ options && options.extensionLocation }
      theme={ options && options.theme }
      typeOfExtension={ options && options.typeOfExtension }
      ongoingSessionExists={ options && options.ongoingSessionExists }
      channelIsWhitelistedForSumo={ options && options.channelIsWhitelistedForSumo }
      onStartGameSession={ options && options.onStartGameSession }
      onEndGameSession={ options && options.onEndGameSession }
    />
  );
}

const setupShallow = (options?: PropOptions) => {
  return shallow(getDefaultProps(options));
}

describe("Config root presentation component", () => {
  it("renders in component mode", () => {
    const component = setupShallow({ typeOfExtension: COMPONENT });
    expect(component.find(CoreText)).toHaveLength(4);
    expect(component.find(StreamLocationPreview)).toHaveLength(0);
  });

  it("renders in overlay mode", () => {
    const component = setupShallow({ typeOfExtension: OVERLAY });
    expect(component.find(CoreText)).toHaveLength(3);
    expect(component.find(StreamLocationPreview)).toHaveLength(1);
    expect(component.find(Dropdown)).toHaveLength(1);
  });

  it("renders an ErrorMessage component in overlay mode when there exists an error message.", () => {
    const component = setupShallow({ typeOfExtension: OVERLAY, errorMessage: "errorMessage" });
    expect(component.find(ErrorMessage)).toHaveLength(1);
  });

  it("does not render an ErrorMessage component in overlay mode when there is no error message.", () => {
    const component = setupShallow({ typeOfExtension: OVERLAY, errorMessage: null });
    expect(component.find(ErrorMessage)).toHaveLength(0);
  });

  it("does not render either game start or end buttons when in component mode.", () => {
    const component = setupShallow({ ongoingSessionExists: false, channelIsWhitelistedForSumo: true, typeOfExtension: COMPONENT });
    expect(component.find({ 'data-test-selector': TestSelectors.GameStartSessionButton })).toHaveLength(0);
    expect(component.find({ 'data-test-selector': TestSelectors.GameEndSessionButton })).toHaveLength(0);
  });

  it("renders the game start session button when there is no ongoing session in overlay mode.", () => {
    const component = setupShallow({ ongoingSessionExists: false, channelIsWhitelistedForSumo: true, typeOfExtension: OVERLAY });
    expect(component.find({ 'data-test-selector': TestSelectors.GameStartSessionButton })).toHaveLength(1);
    expect(component.find({ 'data-test-selector': TestSelectors.GameEndSessionButton })).toHaveLength(0);
  });

  it("renders the game end session button when there is an ongoing session in overlay mode.", () => {
    const component = setupShallow({ ongoingSessionExists: true, channelIsWhitelistedForSumo: true, typeOfExtension: OVERLAY });
    expect(component.find({ 'data-test-selector': TestSelectors.GameStartSessionButton })).toHaveLength(0);
    expect(component.find({ 'data-test-selector': TestSelectors.GameEndSessionButton })).toHaveLength(1);
  });

  it("does not render either game start or end buttons when the channel is not whitelisted for the game.", () => {
    let component = setupShallow({ ongoingSessionExists: true, channelIsWhitelistedForSumo: false, typeOfExtension: OVERLAY });
    expect(component.find({ 'data-test-selector': TestSelectors.GameStartSessionButton })).toHaveLength(0);
    expect(component.find({ 'data-test-selector': TestSelectors.GameEndSessionButton })).toHaveLength(0);

    component = setupShallow({ ongoingSessionExists: false, channelIsWhitelistedForSumo: false, typeOfExtension: OVERLAY });
    expect(component.find({ 'data-test-selector': TestSelectors.GameStartSessionButton })).toHaveLength(0);
    expect(component.find({ 'data-test-selector': TestSelectors.GameEndSessionButton })).toHaveLength(0);
  });
});