import * as React from "react";
import { Location } from "core/types/CommunityService";
import { OVERLAY } from "core/store/modules/Display";
import { Dropdown } from "live_config/components/Dropdown";
import { StreamLocationPreview } from "live_config/components/StreamLocationPreview";
import "./styles.scss";
import { AlignContent, Background, Button, ButtonType, Color, CoreText, Display, Layout, StyledLayout, TextType, FlexDirection, JustifyContent } from "twitch-core-ui";
import { ErrorMessage } from "live_config/components/ErrorMessage";

export interface Props {
  errorMessage: string;
  extensionLocation: Location;
  theme: string;
  typeOfExtension: string;
  ongoingSessionExists: boolean;
  channelIsWhitelistedForSumo: boolean;
  onStartGameSession: () => void;
  onEndGameSession: () => void;
}

export enum TestSelectors {
  GameStartSessionButton = 'GameStartSessionButton',
  GameEndSessionButton = 'GameEndSessionButton',
}

export class RootComponent extends React.Component<Props> {

  public render() {
    return (
      <StyledLayout 
        className={ `tw-theme--${this.props.theme}` }
      >
        <StyledLayout 
          background={ Background.Base }
          color={ Color.Base }
          padding={ 1 }
          className={ 'live-config' }
        >
        { this.props.errorMessage && <ErrorMessage message={this.props.errorMessage} /> } 
        {this.renderGameSection()}
        {this.renderExtensionLocationSection()}
        </StyledLayout>
      </StyledLayout>
    );
  }

  private renderGameSection = () => {
    if (this.props.channelIsWhitelistedForSumo) {
      return (
        <div className="live-config__section">
          <CoreText className="live-config__section__header">Extension Minigame</CoreText>
          { this.renderExtensionGameControls() }
        </div>
      );
    }
    
    return;
  }

  private renderExtensionLocationSection = () => {
    return (
        <div className="live-config__section">
          <CoreText className="live-config__section__header">Extension Location</CoreText>
          { this.renderConfigurationOptions() }
        </div>
    );
  }

  private renderConfigurationOptions = () => {
    if (this.props.typeOfExtension === OVERLAY) {
      const extensionLocation = this.props.extensionLocation;
      return (
        <StyledLayout 
          alignContent={ AlignContent.Center }
          display={ Display.Flex }
          flexDirection={ FlexDirection.Column }
          fullHeight
        >
          <div className="live-config__section__instructions">
            <CoreText className="live-config__section__section-explanation">
              Where would you like to put your Prime Subscription and Loot Reminder overlay?
            </CoreText>
            <CoreText className="live-config__note">
              Note: viewers will need to refresh the stream to see your changes.
            </CoreText>
          </div>
          <StyledLayout 
            alignContent={ AlignContent.Center }
            display={ Display.Flex }
            flexDirection={ FlexDirection.Column }
            justifyContent={ JustifyContent.Center }
            className="live-config__location-selection"
          >
            <StreamLocationPreview extensionLocation={ extensionLocation } />
            <Dropdown />
          </StyledLayout>
        </StyledLayout>
      )
    } else {
      return (
        <div>
          <CoreText type={TextType.P} className="live-config__paragraph">
            The Prime Subscription and Loot Reminder extension is currently in Component mode.
          </CoreText>
          <CoreText type={TextType.P} className="live-config__paragraph">
            It will appear on the right side of your stream, along with other extensions in Component mode.
          </CoreText>
          <CoreText type={TextType.P} className="live-config__paragraph">
            You can click the extension's cog icon (in Dashboard > Extensions > My Extensions) to follow the configuration instructions.
          </CoreText>
        </div>
      )
    }
  }

  private renderExtensionGameControls() {
    if (this.props.typeOfExtension === OVERLAY) {
      if (this.props.ongoingSessionExists == null) {
        return (
            <CoreText 
              className="live-config__section__instructions"
            >
              Loading Prime Gaming Sumo...
            </CoreText>
        )
      }
      if (this.props.ongoingSessionExists) {
        return (
          <React.Fragment>
            <CoreText 
              className="live-config__section__instructions"
              data-test-selector={TestSelectors.GameEndSessionButton}
            >
              The Prime Gaming Sumo minigame is enabled on your channel. Direct your viewers to play by clicking the "Prime Gaming Sumo" icon in the stream window.
            </CoreText>
            <Layout display={Display.Flex}>
              <Layout fullWidth>
                <Button
                  fullWidth
                  type={ButtonType.Hollow}
                  onClick={this.onEndGameSessionClick}
                >
                Disable Prime Gaming Sumo
                </Button>
              </Layout>
            </Layout>
          </React.Fragment>
        )
      } else {
        return (
          <React.Fragment>
            <CoreText 
              className="live-config__section__instructions" 
              data-test-selector={TestSelectors.GameStartSessionButton}
            >
              You can enable a minigame on your channel for your viewers to play against each other. Viewers will be able to keep playing until you disable the minigame, with games lasting no more than two minutes.
            </CoreText>
            <Layout display={Display.Flex}>
              <Layout fullWidth>
                <Button
                  fullWidth
                  onClick={this.onStartGameSessionClick}
                >
                  Enable Prime Gaming Sumo
                </Button>
              </Layout>
            </Layout>
          </React.Fragment>
        )
      }
    } else {
      return (
        <CoreText
          className="live-config__section__instructions"
        >
          The extension minigame only works when this extension is in Overlay mode.
        </CoreText>
      )
    }
  }

  private onStartGameSessionClick = () => {
    this.props.onStartGameSession();
  }

  private onEndGameSessionClick = () => {
    this.props.onEndGameSession();
  }
}