import "./styles.scss";
import * as React from "react";
import * as ReactCSSTransitionGroup from "react-addons-css-transition-group";
import { DismissButton } from "viewer/components/DismissButton";
import { DropdownButton } from "viewer/components/DropdownButton";
import { Error } from "viewer/components/Error";
import { Location } from 'core/types/CommunityService'
import { OVERLAY } from "core/store/modules/Display";
import { Offer } from "viewer/components/Offer";
import { ExtensionGame } from "extension_game";
import { Subscription } from "viewer/components/Subscription";
import { Upsell } from "viewer/components/Upsell";

const COLLAPSED_TIMEOUT = 1000; // One second.

// Props shape.
export interface Props {
  communityServiceLoaded: boolean;
  error: boolean;
  isDismissed: boolean;
  isExpanded: boolean;
  isPrime: boolean;
  typeOfExtension: string;
  showGameLauncher: boolean;
  reportGameLaunch: () => void;
  location: Location;
  displayAsSidenavPopup?: boolean;
}

// State shape.
export interface State {
  show: boolean;
  showDismissButton: boolean;
  showGame: boolean;
}

// React component.
export class RootComponent extends React.PureComponent<Props, State> {

  // Initial state.
  public state: State = {
    show: false,
    showDismissButton: false,
    showGame: false,
  };

  private hideTimeoutId: number;

  public componentDidMount() {
    this.handleHiding();
  }

  public render() {
    if (this.props.communityServiceLoaded || this.props.error) {
      // The prime extension needs to render as a sidenav popup with the 07/2019 component redesign.
      // See: https://discuss.dev.twitch.tv/t/changes-coming-to-component-extensions-and-how-to-update/21201
      if (this.props.displayAsSidenavPopup) {
        return this.renderSidenavPopupExtension();
      } else if (!this.props.isDismissed) {
        return this.renderRegularExtension();
      }
    }

    return null;
  }

  private renderSumoGame() {
    if (this.state.showGame) {
      return (
        <ExtensionGame closeGame={this.closeGame}/>
      );
    } else {
      return null;
    }
  }

  private renderGameIcon = () => {
    return (
      <img 
        className={"prime-extension-root__game__icon"} 
        alt={"game icon"} 
        src="https://m.media-amazon.com/images/G/01/sm/reminder-extension/sumo/sumo_launcher_icon._CB418303477_.png"  
        onClick={this.launchGame}
      />
    )
  }

  private renderGame() {
    return (
      <React.Fragment>
        <ReactCSSTransitionGroup
          transitionName="sumo-game"
          transitionEnterTimeout={600}
          transitionLeaveTimeout={300}
        >
          {this.renderSumoGame()}
        </ReactCSSTransitionGroup>
        {this.props.showGameLauncher && this.props.typeOfExtension === OVERLAY && this.state.show && !this.state.showGame && this.renderGameIcon()}
      </React.Fragment>
    );
  }

  private renderMainExtension() {
    if (this.state.showGame) {
      return null;
    } else {
      let locationModifier = this.LocationModifiers[this.props.location];
      return (
        <div>
          <ReactCSSTransitionGroup
            transitionName="prime-extension-root"
            transitionEnterTimeout={300}
            transitionLeaveTimeout={300}
          >
            <div
              className={`prime-extension-root  
                prime-extension-root--${this.state.show || this.props.isExpanded ? "show" : "hide"}`
              }
              key="prime-extension-root"
              onMouseMove={this.markActive}
              onMouseLeave={this.markInactive}
            >
              <div className={`prime-extension-root__extension${this.props.typeOfExtension === OVERLAY ? ` prime-extension-root__extension--${locationModifier}` : " prime-extension-root__extension--component" }`}>
                {(this.state.showDismissButton || this.props.isExpanded) &&
                  <div className="prime-extension-root__extension__dismiss">
                    <DismissButton />
                  </div>
                }
                <DropdownButton />
                <ReactCSSTransitionGroup
                  transitionName="prime-expanded-section"
                  transitionEnterTimeout={600}
                  transitionLeaveTimeout={300}
                >
                  {this.props.isExpanded && this.renderExpandedSection()}
                </ReactCSSTransitionGroup>
              </div>
            </div>
          </ReactCSSTransitionGroup>
        </div>
      );
    }
  }

  private renderSidenavPopupExtension = () => {
    return (
      <div className="prime-extension-sidenav-popup">
        {this.renderExpandedSection()}
      </div>
    );
  }

  private renderRegularExtension = () => {
    return (
      <div  
        className={`user-watcher ${this.state.showGame? 'prime-extension-root--darkened' : ''}`}
        onMouseEnter={this.show}
        onMouseMove={this.show}
        onMouseLeave={this.hide}
      >
        {this.renderGame()}
        {this.renderMainExtension()}
      </div>
    );
  } 

  private renderExpandedSection = () => {
    if (this.props.error) {
      return (
        <div key="prime-expanded-section">
          <Error />
        </div>
      );
    } else {
      return (
        <div key="prime-expanded-section">
          { this.props.isPrime ? <Subscription /> : <Upsell /> }
          <Offer />
        </div>
      )
    }
  }

  private launchGame = () => {
    this.props.reportGameLaunch();
    this.setState({showGame: true});
  }

  private closeGame = () => {
    this.setState({showGame: false});
  }

  private show = () => {
    this.setState({ show: true });
    this.resetHideTimeout();
  }

  private hide = () => {
    this.setState({ show: false });
    this.clearHideTimeout();
  }

  private markActive = (e) => {
    e.stopPropagation();
    this.clearHideTimeout();
    this.setState({ showDismissButton: true });
  }

  private markInactive = () => {
    this.resetHideTimeout();
    this.setState({ showDismissButton: false });
  }

  private handleHiding = () => {
    if (this.props.isExpanded) {
      this.clearHideTimeout();
    } else {
      this.resetHideTimeout();
    }
  }

  private resetHideTimeout = () => {
    this.clearHideTimeout();
    this.hideTimeoutId = window.setTimeout(this.hide, COLLAPSED_TIMEOUT);
  }

  private clearHideTimeout = () => {
    clearTimeout(this.hideTimeoutId);
  }

  LocationModifiers = {
    TOP_LEFT : "top-left",
    TOP_MIDDLE : "top-middle",
    TOP_RIGHT : "top-right",
    BOTTOM_LEFT : "bottom-left",
    BOTTOM_RIGHT : "bottom-right"
  } 
}
