import { LootType } from "core/types/CommunityService";
import { shallow } from "enzyme";
import * as React from "react";
import { DropdownButtonCssClass, 
  DropdownButtonComponent, 
  DROPDOWN_BUTTON_CROWN, 
  DropdownButtonMessage,
  DROPDOWN_BUTTON_NON_EXPANDED, 
  DROPDOWN_BUTTON_TEXT, 
  getDropdownCrownCssClass } from "viewer/components/DropdownButton/component";

interface OptionProps {
  isPrime?: boolean;
  canSubscribe?: boolean;
  canSubscribeToStreamer?: boolean;
  type?: LootType;
  error?: boolean;
  subscribed?: boolean;
  isSubscribing?: boolean;
  toggle?: () => {};
  isExpanded?: boolean;
}

const getDropdownButtonComponent = (options: OptionProps) => {
  return (
    <DropdownButtonComponent
        isPrime={options && options.isPrime}
        canSubscribe={options && options.canSubscribe}
        canSubscribeToStreamer={options && options.canSubscribeToStreamer}
        type={options && options.type || LootType.LOOT}
        error={options && options.error}
        subscribed={options && options.subscribed}
        isSubscribing={options && options.isSubscribing || false}
        toggle={jest.fn()}
        isExpanded={options && options.isExpanded || false}
    />
  );
}

const setupShallow = (options?: OptionProps) => {
  return shallow(getDropdownButtonComponent(options));
}

const testDropdownButtonContents = (component, message: DropdownButtonMessage, cssClass: DropdownButtonCssClass) => {
  expect(component.find(`.${cssClass}`).exists()).toBe(true);
  expect(component.find(`.${DROPDOWN_BUTTON_CROWN}`).hasClass(getDropdownCrownCssClass(cssClass))).toBe(true);
  expect(component.find(`.${DROPDOWN_BUTTON_TEXT}`).text()).toEqual(message);
}

document.getElementById = (id) => {
    return document.createElement("div");
};

describe("Dropdownbutton presentation component", () => {
  it("renders a free loot message if the sub is not available", () => {
    const component = setupShallow({ isPrime: true, canSubscribeToStreamer: true, isExpanded: false });
    testDropdownButtonContents (component, DropdownButtonMessage.FreeLoot, DropdownButtonCssClass.FreeStuff);
  });

  it("renders a free game message if the sub is not available and the content is a game", () => {
    const component = setupShallow({ isPrime: true, canSubscribeToStreamer: true, type: LootType.GAME });
    testDropdownButtonContents (component, DropdownButtonMessage.FreeGame, DropdownButtonCssClass.FreeStuff);
   });

  it("renders a free loot message if the sub is available but the user cannot subscribe", () => {
    const component = setupShallow({ isPrime: true, canSubscribe: true });
    testDropdownButtonContents (component, DropdownButtonMessage.FreeLoot, DropdownButtonCssClass.FreeStuff);
  });

  it("renders a free sub ready message if the sub is available and the user can subscribe", () => {
    const component = setupShallow({ isPrime: true, canSubscribe: true, canSubscribeToStreamer: true });
    testDropdownButtonContents (component, DropdownButtonMessage.FreeSub, DropdownButtonCssClass.FreeSub);
  });

  it("renders a free sub ready message if the user is in the middle of the subscription animation", () => {
    const component = setupShallow({ isPrime: true, isSubscribing: true });
    testDropdownButtonContents (component, DropdownButtonMessage.FreeSub, DropdownButtonCssClass.FreeSub);
  });

  it("renders a subscribed message if the user has subscribed during this session", () => {
    const component = setupShallow({ isPrime: true, canSubscribeToStreamer: true, isExpanded: true, subscribed: true });
    testDropdownButtonContents (component, DropdownButtonMessage.Subscribed, DropdownButtonCssClass.Subscribed);
  });

  it("renders an error message if there was an error", () => {
    const component = setupShallow({ isPrime: true, canSubscribeToStreamer: true, isExpanded: true, error: true, subscribed: true });
    testDropdownButtonContents (component, DropdownButtonMessage.Error, DropdownButtonCssClass.Error);
  });

  it("renders the dropdown with a gradient when the user is not logged in", () => {
    const component = setupShallow({isExpanded: true});
    expect(component.find(`.${DROPDOWN_BUTTON_NON_EXPANDED}`).exists()).toBe(true);
  });

});
