import { LootType } from "core/types/CommunityService";
import * as React from "react";
import "./styles.scss";


export enum DropdownButtonMessage {
  FreeLoot = "Free Loot",
  FreeGame = "Free Game",
  Error = "Uh oh",
  FreeSub = "Free Sub Ready",
  Subscribed = "Subscribed",
}

export enum DropdownButtonCssClass {
  FreeStuff = "dropdown-button--free-stuff",
  Error = "dropdown-button--error",
  FreeSub = "dropdown-button--sub-ready",
  Subscribed = "dropdown-button--subscribed",
}

export function getDropdownCrownCssClass(cssClass: DropdownButtonCssClass) {
  return cssClass + "__crown";
}

export const DROPDOWN_BUTTON_CROWN = "dropdown-button__crown";
export const DROPDOWN_BUTTON_TEXT = "dropdown-button__text";
export const DROPDOWN_BUTTON_NON_EXPANDED = "dropdown-button--non-prime-expanded";

export interface Props {
  isPrime: boolean;
  canSubscribe: boolean;
  canSubscribeToStreamer: boolean;
  error: boolean;
  isExpanded: boolean;
  subscribed: boolean;
  isSubscribing: boolean;
  toggle: () => void;
  type: LootType;
  opaqueUserId?: string;
}

export class DropdownButtonComponent extends React.Component<Props> {
    private dropdownButton;

    constructor(props) {
      super(props);
      this.dropdownButton = React.createRef();
    }

    public componentDidMount() {
      const dropdownButton = this.dropdownButton;
      const dropdownButtonDomElement = dropdownButton && dropdownButton.current;
      dropdownButtonDomElement && dropdownButtonDomElement.addEventListener("dblclick", this.stopPropagatingDoubleClick);
    }

    public componentWillUnmount() {
      const dropdownButton = this.dropdownButton;
      const dropdownButtonDomElement = dropdownButton && dropdownButton.current;
      dropdownButtonDomElement && dropdownButtonDomElement.removeEventListener("dblclick", this.stopPropagatingDoubleClick);
    }

    public render() {
      let message = this.props.type === LootType.LOOT ? DropdownButtonMessage.FreeLoot : DropdownButtonMessage.FreeGame;
      let cssClass = DropdownButtonCssClass.FreeStuff;
      if (this.props.error) {
        message = DropdownButtonMessage.Error;
        cssClass = DropdownButtonCssClass.Error;
      } else if (this.props.isPrime && ((this.props.canSubscribe && this.props.canSubscribeToStreamer) || this.props.isSubscribing)) {
        message = DropdownButtonMessage.FreeSub;
        cssClass = DropdownButtonCssClass.FreeSub;
      } else if (this.props.isPrime && this.props.subscribed) {
        message = DropdownButtonMessage.Subscribed;
        cssClass = DropdownButtonCssClass.Subscribed;
      }
      const nonPrimeExpanded = this.props.isExpanded && !this.props.isPrime;
      const nonPrimeExpandedClass = nonPrimeExpanded ? DROPDOWN_BUTTON_NON_EXPANDED : "";
      return (
        <div ref={ this.dropdownButton }
         className={ `dropdown-button ${cssClass} ${nonPrimeExpandedClass}` }
         onClick={ this.props.toggle }
        >
        {this.props.opaqueUserId && <img className="dropdown-button__pixel" src={"https://api.twitch.tv/v5/amazon/session/track?source=prime_extension&session_id=" + this.props.opaqueUserId + "&client_id=d4uvtfdr04uq6raoenvj7m86gdk16v"}/> }
          <div>
            <span className={`${DROPDOWN_BUTTON_CROWN} ${getDropdownCrownCssClass(cssClass)}`}>
              <img className={DROPDOWN_BUTTON_CROWN + "__icon"} src="https://m.media-amazon.com/images/G/01/sm/reminder-extension/crown_White_72x72.svg" alt="logo"></img>
            </span>
            <span className={DROPDOWN_BUTTON_TEXT}>
              {message}
            </span>
          </div>
          <span className="dropdown-button__arrow">
            <img src="https://m.media-amazon.com/images/G/01/sm/reminder-extension/dropdown-arrow-hd._CB480069040_.png" />
          </span>
        </div>
      );
    }

    private stopPropagatingDoubleClick(e) {
      e.stopPropagation();
    }
}
