import * as React from "react";
import { Animation,
  AnimationDuration,
  AnimationType,
  CoreText,
  Display,
  Layout,} from "twitch-core-ui";
import "./styles.scss";
import { getTypeOfExtension } from "core/utils/extensionSettings";

export const SUBSCRIBE_BUTTON = "subscription__subscribe-button";
export const SUBSCRIBED_BANNER = "subscription--subscribed";
export const TYPE_OF_EXTENSION = getTypeOfExtension();

export interface Props {
  hasCredit: boolean;
  canSubscribeToStreamer: boolean;
  streamerName: string;
  subscribed: boolean;
  isSubscribing: boolean;
  subscribedTime: string;
  subscribedChannel: string;
  onSubscribeClick: () => void;
  setSubscribingStatus: (status: boolean) => void;
}

export interface State {
  crownCentered: boolean;
}

export class SubscriptionComponent extends React.Component<Props, State> {
  // Initial state.
  public state: State = {
    crownCentered: false
  };

  private subscriptionButton;

  constructor(props) {
    super(props);
    this.subscriptionButton = React.createRef();
  }

  public componentDidMount() {
    const subscriptionButton = this.subscriptionButton;
    const subscriptionButtonDomElement = subscriptionButton && subscriptionButton.current;
    subscriptionButtonDomElement && subscriptionButtonDomElement.addEventListener("dblclick", this.stopPropagatingDoubleClick);
  }

  public componentWillUnmount() {
    const subscriptionButton = this.subscriptionButton;
    const subscriptionButtonDomElement = subscriptionButton && subscriptionButton.current;
    subscriptionButtonDomElement && subscriptionButtonDomElement.removeEventListener("dblclick", this.stopPropagatingDoubleClick);
  }

  public render() {

    // If the user cannot subscribe and has no active subscription, return null.
    if (!this.haveCurrentSubscription()
      && !this.props.subscribed
      && !this.canSubscribe()
    ) {
      return null;
    }

    const subscriptionClassNamePrefix = "subscription";
    // If the user has an active subscription, display the subscription.  Unless their free prime sub is
    // available and they can subscribe to the current streamer.  Then show that instead.
    if (this.haveCurrentSubscription() && !this.canSubscribe() && !this.props.isSubscribing) {
      let date = new Date(this.props.subscribedTime)
        .toLocaleDateString("en-US", {month: "short", day: "numeric", year: "numeric"});
      return (
        <Layout className={`${subscriptionClassNamePrefix} subscription--subscribed`}>
          <CoreText className={`subscription__streamer-name subscription__streamer-name_font--${TYPE_OF_EXTENSION}`}>
            {this.props.subscribedChannel}
          </CoreText>
          <Layout className="subscription__date" fullWidth>
            <CoreText>
              {date}
            </CoreText>
          </Layout>
        </Layout>
      );
    }

    // Display the subscription button.
    const subscriptionClassName = `${subscriptionClassNamePrefix} subscription--${!this.props.hasCredit && !this.props.isSubscribing ? "subscribed" : "subscribing"}`;
    const date = new Date();
    const dateString = date.toLocaleDateString("en-US", {month: "short", day: "numeric", year: "numeric"});
    const buttonClassName = `subscription__subscribe-button subscription__subscribe-button__button_size--${TYPE_OF_EXTENSION} subscription__subscribe-button--${this.props.isSubscribing ? "subscribing" : "waiting"}`;

    return (
      <Layout className={subscriptionClassName}>
        <CoreText className={`subscription__streamer-name subscription__streamer-name_font--${TYPE_OF_EXTENSION}`}>
          {this.props.streamerName}
        </CoreText>
        <Animation
          type={AnimationType.FadeOut}
          duration={AnimationDuration.Medium}
          enabled={!this.props.hasCredit && !this.props.isSubscribing}
        >
          <div
            ref={ this.subscriptionButton }
            className={buttonClassName}
            onClick={this.props.isSubscribing ? () => {} : this.onSubscribe}
            data-test-target={SUBSCRIBE_BUTTON}
          >
            <Layout className="subscription__subscribe-button__crown-wrapper" display={Display.Inline}>
              <div className={`subscription__subscribe-button__crown-wrapper__crown subscription__subscribe-button__crown-wrapper__svg--${TYPE_OF_EXTENSION}`}>
                <img className="subscription__subscribe-button__crown-wrapper__crown__icon" src="https://m.media-amazon.com/images/G/01/sm/reminder-extension/crown_White_72x72.svg" alt="logo"></img>
              </div>
            </Layout>
            <Animation
              type={AnimationType.FadeOut}
              duration={AnimationDuration.Medium}
              enabled={this.props.isSubscribing}
              onAnimationEnd={this.centerCrown}
            >
              <Layout className={`subscription__subscribe-button__text subscription__subscribe-button__text_font--${TYPE_OF_EXTENSION}`}
               display={Display.Inline}>Subscribe Free</Layout>
            </Animation>
          </div>
        </Animation>
        <Animation
          type={AnimationType.FadeIn}
          duration={AnimationDuration.Medium}
          enabled={!this.props.hasCredit && !this.props.isSubscribing}
        >
          <Layout className= {`subscription__date subscription__date_font--${TYPE_OF_EXTENSION}`} fullWidth>
            <CoreText>
              {dateString}
            </CoreText>
          </Layout>
        </Animation>
      </Layout>
    );
  }

  private haveCurrentSubscription() {
    return this.props.subscribedChannel && this.props.subscribedTime;
  }

  private canSubscribe() {
    return this.props.hasCredit && this.props.canSubscribeToStreamer;
  }

  private stopPropagatingDoubleClick(e) {
    e.stopPropagation();
  }

  private onSubscribe = () => {
    this.props.setSubscribingStatus(true);
    this.props.onSubscribeClick();
  }

  private centerCrown = () => {
    setTimeout(() => {
      this.setState({ crownCentered: true });
      this.resetCrown();
    }, 300);
  }

  private resetCrown = () => {
    setTimeout(() => {
      this.setState({ crownCentered: false });
      this.props.setSubscribingStatus(false);
    }, 2000);
  }
}
