package api

import (
	"net/http"

	log "github.com/sirupsen/logrus"

	"github.com/cactus/go-statsd-client/statsd"

	rdrpc "code.justin.tv/samus/red-dot/rpc"

	"time"

	"code.justin.tv/foundation/twitchserver"
	"code.justin.tv/samus/red-dot/config"
	"code.justin.tv/samus/red-dot/metrics"
	"code.justin.tv/samus/red-dot/middleware"
	"goji.io"
	"goji.io/pat"
)

// RedDotServer API
type RedDotServer struct {
	OfferStatusAPI
}

// Server struct
type Server struct {
	*goji.Mux
}

// NewServer defines handlers
func (n *RedDotServer) NewServer(stats statsd.Statter, cfg *config.Configuration) (*Server, error) {
	server := twitchserver.NewServer()

	twitchserver.AddDefaultSignalHandlers()
	s := &Server{
		server,
	}

	metricsConfig := metrics.Config{
		Stage:                          cfg.Environment,
		AwsRegion:                      cfg.AWSRegion,
		BufferSize:                     cfg.CloudwatchBufferSize,
		BatchSize:                      cfg.CloudwatchBatchSize,
		FlushInterval:                  time.Second * time.Duration(cfg.CloudwatchFlushIntervalSeconds),
		FlushPollCheckDelay:            time.Millisecond * time.Duration(cfg.CloudwatchFlushCheckDelayMs),
		BufferEmergencyFlushPercentage: cfg.CloudwatchEmergencyFlushPercentage,
	}

	metricLogger, metricsFlusher := metrics.New(metricsConfig)
	metricsMiddleware := middleware.NewMetricsMiddleware(metricLogger)
	go metricsFlusher.FlushMetricsAtInterval()

	// Add Metrics and Panic Recover Middleware
	s.Use(middleware.AccessLogger)
	s.Use(metricsMiddleware.Metrics)

	s.HandleFunc(pat.Get("/"), s.everythingIsFine)
	s.HandleFunc(pat.Get("/health"), s.everythingIsFine)
	s.HandleFunc(pat.Get("/debug/running"), s.everythingIsFine)

	twirpHandler := rdrpc.NewRedDotServer(n, nil, nil)
	s.Handle(pat.Post(rdrpc.RedDotPathPrefix+"*"), twirpHandler)

	return s, nil
}

func (s *Server) everythingIsFine(w http.ResponseWriter, r *http.Request) {
	_, err := w.Write([]byte("OK"))
	if err != nil {
		log.Fatal(err)
	}
}
