package metrics

import (
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/cloudwatch"
	"github.com/aws/aws-sdk-go/service/cloudwatch/cloudwatchiface"
	log "github.com/sirupsen/logrus"
)

const (
	redDotNamespace = "red-dot"
	STAGE           = "STAGE"
	COUNT           = "Count"
)

type Config struct {
	Stage                          string
	AwsRegion                      string
	BufferSize                     int
	BatchSize                      int
	FlushInterval                  time.Duration
	FlushPollCheckDelay            time.Duration
	BufferEmergencyFlushPercentage float64
}

// RedDotMetricLogger wrapper struct that uses Cloudwatch to post Metrics data
type RedDotMetricLogger struct {
	metricsBuffer chan *cloudwatch.MetricDatum
	config        Config
}

// RedDotMetricFlusher struct containing fields required to run metric flushing
type RedDotMetricFlusher struct {
	cloudWatch    cloudwatchiface.CloudWatchAPI
	metricsBuffer chan *cloudwatch.MetricDatum
	config        Config
	lastFlushTime time.Time
}

// IMetricLogger API for creating metrics from the service
type IMetricLogger interface {
	// LogDurationMetric captures a metric duration of time
	LogDurationMetric(name string, duration time.Duration)
	// LogDurationSinceMetric captures the duration since a given event timestamp
	LogDurationSinceMetric(name string, timeOfEvent time.Time)
	// LogCountMetric captures count metrics
	LogCountMetric(name string, count float64)
}

// NilMetricsLogger equivalent of NullMetricsFactory
type NilMetricsLogger struct{}

func (n *NilMetricsLogger) LogDurationMetric(name string, duration time.Duration) {
	return
}
func (n *NilMetricsLogger) LogDurationSinceMetric(name string, timeOfEvent time.Time) {
	return
}
func (n *NilMetricsLogger) LogCountMetric(name string, count float64) {
	return
}

// NewNilMetricsLogger creates a NilMetricsLogger
func NewNilMetricsLogger() IMetricLogger {
	return &NilMetricsLogger{}
}

// IMetricFlusher interface for flushing metrics
type IMetricFlusher interface {
	FlushMetrics()
	FlushMetricsAtInterval()
	ShouldFlush() bool
}

// New creates an implementation of IMetricLogger from the Metric Config
func New(config Config) (IMetricLogger, IMetricFlusher) {
	awsConfig := &aws.Config{
		Region: aws.String(config.AwsRegion),
	}
	client := cloudwatch.New(session.New(), awsConfig)

	return NewFromCloudwatchClient(config, client)
}

// NewFromCloudwatchClient creates an IMetricLogger using CW Client
func NewFromCloudwatchClient(config Config, cloudwatchClient cloudwatchiface.CloudWatchAPI) (IMetricLogger, IMetricFlusher) {
	metricsBufferChannel := make(chan *cloudwatch.MetricDatum, config.BufferSize)

	logger := &RedDotMetricLogger{
		config:        config,
		metricsBuffer: metricsBufferChannel,
	}

	flusher := &RedDotMetricFlusher{
		cloudWatch:    cloudwatchClient,
		metricsBuffer: metricsBufferChannel,
		config:        config,
		lastFlushTime: time.Now(),
	}

	return logger, flusher
}

// LogDurationSinceMetric captures the duration since a given event timestamp
func (this *RedDotMetricLogger) LogDurationSinceMetric(name string, timeOfEvent time.Time) {
	delta := time.Since(timeOfEvent)
	this.LogDurationMetric(name, delta)
}

// LogDurationMetric captures a metric duration of time
func (this *RedDotMetricLogger) LogDurationMetric(name string, duration time.Duration) {
	metricDatum := &cloudwatch.MetricDatum{
		MetricName: aws.String(name),
		Dimensions: []*cloudwatch.Dimension{
			{
				Name:  aws.String("STAGE"),
				Value: aws.String(this.config.Stage),
			},
		},
		Timestamp: aws.Time(time.Now().UTC()),
		Unit:      aws.String("Seconds"),
		Value:     aws.Float64(duration.Seconds()),
	}
	if len(this.metricsBuffer) < this.config.BufferSize {
		this.metricsBuffer <- metricDatum
	} else {
		log.Error("Metrics buffer at capacity. Additional metrics will be dropped")
	}
}

// LogCountMetric captures count metrics
func (this *RedDotMetricLogger) LogCountMetric(name string, count float64) {
	metricDatum := &cloudwatch.MetricDatum{
		MetricName: aws.String(name),
		Dimensions: []*cloudwatch.Dimension{
			{
				Name:  aws.String(STAGE),
				Value: aws.String(this.config.Stage),
			},
		},
		Timestamp: aws.Time(time.Now().UTC()),
		Unit:      aws.String(COUNT),
		Value:     aws.Float64(count),
	}
	if len(this.metricsBuffer) < this.config.BufferSize {
		this.metricsBuffer <- metricDatum
	} else {
		log.Error("Metrics buffer at capacity. Additional metrics will be dropped")
	}
}

func (this *RedDotMetricFlusher) FlushMetricsAtInterval() {
	this.lastFlushTime = time.Now()
	for {
		// Start the polling delay timer
		timer := time.After(this.config.FlushPollCheckDelay)

		if this.ShouldFlush() {
			this.FlushMetrics()
			this.lastFlushTime = time.Now()
		}

		// Wait until the polling delay timer expires
		<-timer
	}
}

func (this *RedDotMetricFlusher) ShouldFlush() bool {
	return this.isNextFlushInterval() || this.isBufferApproachingCapacity()
}

func (this *RedDotMetricFlusher) isNextFlushInterval() bool {
	timeSinceLastFlush := time.Now().Sub(this.lastFlushTime)
	return timeSinceLastFlush > this.config.FlushInterval
}

func (this *RedDotMetricFlusher) isBufferApproachingCapacity() bool {
	bufferUsage := float64(len(this.metricsBuffer)) / float64(this.config.BufferSize)
	approachingCapacity := bufferUsage >= this.config.BufferEmergencyFlushPercentage
	if approachingCapacity {
		log.WithField("bufferUsage", bufferUsage).Error("The metrics buffer is approaching capacity. Executing emergency flush.")
	}
	return approachingCapacity
}

func (this *RedDotMetricFlusher) FlushMetrics() {
	numMetricsToFlush := len(this.metricsBuffer)

	numSucceeded, numFailed := 0, 0
	metricBatch := make([]*cloudwatch.MetricDatum, 0, this.config.BatchSize)
	for i := 0; i < numMetricsToFlush; i++ {

		metric := <-this.metricsBuffer
		metricBatch = append(metricBatch, metric)

		// Send the batch if it is at capacity or this is the last metric
		if len(metricBatch) >= this.config.BatchSize || i == (numMetricsToFlush-1) {
			err := this.postMetricBatch(metricBatch)
			if err != nil {
				log.WithError(err).WithField("metricNum", len(metricBatch)).Error("Failed to send metrics while communicating with cloudwatch")
				numFailed += len(metricBatch)
			} else {
				numSucceeded += len(metricBatch)
			}
			metricBatch = make([]*cloudwatch.MetricDatum, 0, this.config.BatchSize)
		}
	}

	if numFailed > 0 {
		log.Infof("Finished flushing metrics, but encountered some errors. Succeeded: [%d], Failed: [%d]", numSucceeded, numFailed)
	}
}

func (this *RedDotMetricFlusher) postMetricBatch(metricsBatch []*cloudwatch.MetricDatum) error {
	request := &cloudwatch.PutMetricDataInput{
		MetricData: metricsBatch,
		Namespace:  aws.String(redDotNamespace),
	}
	_, err := this.cloudWatch.PutMetricData(request)
	return err
}
