package middleware

import (
	"fmt"
	"net/http"
	"strings"

	"code.justin.tv/samus/red-dot/metrics"
	"code.justin.tv/samus/red-dot/rpc"
	"github.com/felixge/httpsnoop"
)

const (
	AVAILABILITY_FORMAT = "%s.Availability"
)

type MetricsMiddleware struct {
	metricLogger metrics.IMetricLogger
}

// NewMetricsMiddleware provides the Logger to capture API metrics on Red Dot
func NewMetricsMiddleware(metricLogger metrics.IMetricLogger) *MetricsMiddleware {
	return &MetricsMiddleware{
		metricLogger: metricLogger,
	}
}

func isTwirpRequest(r *http.Request) bool {
	if r == nil || r.URL == nil {
		return false
	}
	return strings.HasPrefix(r.URL.Path, red_dot.RedDotPathPrefix)
}

func getTwirpMetricName(r *http.Request) string {
	if r == nil || r.URL == nil {
		return ""
	}
	return fmt.Sprintf("TWIRP:%s", strings.TrimPrefix(r.URL.Path, red_dot.RedDotPathPrefix))
}

// Metrics middleware method to get API level metrics (taken from paydays cloudwatch metrics)
func (m *MetricsMiddleware) Metrics(next http.Handler) http.Handler {
	fn := func(w http.ResponseWriter, r *http.Request) {
		metrics := httpsnoop.CaptureMetrics(next, w, r)

		var metricName string
		if isTwirpRequest(r) {
			metricName = getTwirpMetricName(r)
		} else {
			metricName = fmt.Sprintf("%s:%s", r.Method, r.URL)
		}

		m.metricLogger.LogDurationMetric(metricName, metrics.Duration)
		code := metrics.Code
		switch {
		case code < 200:
			m.metricLogger.LogCountMetric(fmt.Sprintf("%s.1xx", metricName), 1.0)
		case code >= 200 && code < 300:
			m.metricLogger.LogCountMetric(fmt.Sprintf("%s.2xx", metricName), 1.0)
		case code >= 300 && code < 400:
			m.metricLogger.LogCountMetric(fmt.Sprintf("%s.3xx", metricName), 1.0)
		case code >= 400 && code < 500:
			m.metricLogger.LogCountMetric(fmt.Sprintf("%s.4xx", metricName), 1.0)
		case code >= 500:
			m.metricLogger.LogCountMetric(fmt.Sprintf("%s.5xx", metricName), 1.0)
		}

		// Availability For API
		m.metricLogger.LogCountMetric(fmt.Sprintf(AVAILABILITY_FORMAT, metricName), float64(calculateAvailability(code)))

	}
	return http.HandlerFunc(fn)
}

// calculateAvailability use status code to
func calculateAvailability(status int) int {
	if status < 500 {
		return 1
	}
	return 0
}
