package middleware_test

import (
	"bytes"
	"io/ioutil"
	"net/http"
	"net/http/httptest"
	"testing"

	"code.justin.tv/samus/red-dot/middleware"
	"code.justin.tv/samus/red-dot/mocks"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
)

func TestMetricsMiddlewareSuccess(t *testing.T) {
	Convey("Test Success when 200 is recorded", t, func() {
		assert := assert.New(t)

		tests := []struct {
			description        string
			url                string
			expectedBody       string
			expectedCode       int
			expectedMetricName string
		}{
			{
				description:        "Metrics Recorded",
				url:                "/health",
				expectedBody:       "OK\n",
				expectedCode:       200,
				expectedMetricName: "GET:/health",
			},
		}

		mockLogger := new(mocks.IMetricLogger)
		metrixer := middleware.NewMetricsMiddleware(mockLogger)
		testServer := httptest.NewServer(metrixer.Metrics(GetOkayTestHandler()))

		for _, tc := range tests {
			mockLogger.On("LogDurationMetric", tc.expectedMetricName, mock.Anything)
			mockLogger.On("LogCountMetric", tc.expectedMetricName+".2xx", 1.0)
			mockLogger.On("LogCountMetric", tc.expectedMetricName+".Availability", 1.0)

			var u bytes.Buffer
			u.WriteString(string(testServer.URL))
			u.WriteString(tc.url)

			res, err := http.Get(u.String())
			assert.NoError(err)
			if res != nil {
				defer res.Body.Close()
			}

			b, err := ioutil.ReadAll(res.Body)
			assert.NoError(err)

			assert.Equal(tc.expectedCode, res.StatusCode, tc.description)
			assert.Equal(tc.expectedBody, string(b), tc.description)
		}

	})
}

func TestMetricsMiddlewareFail(t *testing.T) {
	Convey("Test Success when non-200's are recorded", t, func() {
		assert := assert.New(t)

		tests := []struct {
			description        string
			url                string
			expectedBody       string
			expectedCode       int
			expectedMetricName string
		}{
			{
				description:        "Fail Metrics Recorded",
				url:                "/health",
				expectedBody:       http.StatusText(500) + "\n",
				expectedCode:       500,
				expectedMetricName: "GET:/health",
			},
		}

		mockLogger := new(mocks.IMetricLogger)
		metrixer := middleware.NewMetricsMiddleware(mockLogger)
		testServer := httptest.NewServer(metrixer.Metrics(GetFailedTestHandler()))

		for _, tc := range tests {
			mockLogger.On("LogDurationMetric", tc.expectedMetricName, mock.Anything)
			mockLogger.On("LogCountMetric", tc.expectedMetricName+".5xx", 1.0)
			mockLogger.On("LogCountMetric", tc.expectedMetricName+".Availability", 0.0)

			var u bytes.Buffer
			u.WriteString(string(testServer.URL))
			u.WriteString(tc.url)

			res, err := http.Get(u.String())
			assert.NoError(err)
			if res != nil {
				defer res.Body.Close()
			}

			b, err := ioutil.ReadAll(res.Body)
			assert.NoError(err)

			assert.Equal(tc.expectedCode, res.StatusCode, tc.description)
			assert.Equal(tc.expectedBody, string(b), tc.description)
		}
	})
}

// GetFailedTestHandler returns a http.HandlerFunc for testing http middleware
func GetFailedTestHandler() http.HandlerFunc {
	fn := func(rw http.ResponseWriter, req *http.Request) {
		http.Error(rw, http.StatusText(500), 500)
	}
	return http.HandlerFunc(fn)
}

// GetFailedTestHandler returns a http.HandlerFunc for testing http middleware
func GetOkayTestHandler() http.HandlerFunc {
	fn := func(rw http.ResponseWriter, req *http.Request) {
		rw.Write([]byte("OK\n"))
		rw.WriteHeader(http.StatusOK)
	}
	return http.HandlerFunc(fn)
}
