package main

import (
	"github.com/cactus/go-statsd-client/statsd"

	"time"

	"code.justin.tv/foundation/twitchserver"
	"code.justin.tv/samus/rex/config"
	"code.justin.tv/samus/rex/internal/api"
	"code.justin.tv/samus/rex/internal/app/gifting"
	"code.justin.tv/samus/rex/internal/app/status"
	"code.justin.tv/samus/rex/internal/clients"
	log "github.com/sirupsen/logrus"
)

const (
	prodEnvironment = "prod"
)

func main() {
	environment := config.GetEnvironment()

	logLevel := log.DebugLevel
	if environment == prodEnvironment {
		logLevel = log.InfoLevel
	}
	log.SetLevel(logLevel)
	log.SetFormatter(&log.TextFormatter{FullTimestamp: true, TimestampFormat: time.RFC3339})

	stats, err := statsd.NewNoopClient()
	if err != nil {
		log.WithError(err).Fatal("Failed to initialize statsd client")
	}
	err = startWebServer(stats)
	if err != nil {
		log.Fatalf("Error during server shutdown: %v", err)
	}
}

// Set up the web server start listening for requests.
func startWebServer(stats statsd.Statter) error {
	environment := config.GetEnvironment()
	cfg, err := config.LoadConfigForEnvironment(environment)
	if err != nil {
		log.Fatal(err)
	}

	clients, err := clients.SetupClients(cfg, stats)
	if err != nil {
		log.Fatalf("Failed to initialize clients: %s", err)
	}

	offerStatusAPI := api.OfferStatusAPI{
		OfferStatus: status.NewOfferStatus(*clients),
	}

	giftingAPI := api.GiftingAPI{
		Gifting: gifting.NewGifting(*clients, cfg.GiftingOfferWhitelist, cfg.GiftingOfferWhitelistActive),
	}

	rexServer := api.RexServer{
		OfferStatusAPI: offerStatusAPI,
		GiftingAPI:     giftingAPI,
	}

	server, err := rexServer.NewServer(stats, cfg)
	if err != nil {
		log.Fatal(err)
	}

	log.Info("Starting Rex server")

	serverConfig := twitchserver.NewConfig()
	serverConfig.Statter = stats

	// If you want to run on a port different than 8000, uncomment these lines (useful is running with local visage or other rex consumer)
	// serverConfig.Addr = ":9000"
	// serverConfig.DebugAddr = ":9050"

	twitchserver.AddDefaultSignalHandlers()
	return twitchserver.ListenAndServe(server, serverConfig)
}
