package integration_test

import (
	"context"
	"flag"
	"fmt"
	"log"
	"testing"
	"time"

	appConfig "code.justin.tv/samus/rex/config"
	testConfig "code.justin.tv/samus/rex/integration_test/client"
	rex "code.justin.tv/samus/rex/rpc"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/guregu/dynamo"
	"github.com/pkg/errors"
	. "github.com/smartystreets/goconvey/convey"
)

var _ = func() bool {
	testing.Init()
	return true
}()

// Default test users
const qaSamusTurbo = "207069781"

var offerStatusTable = getOfferStatusTable()

func getOfferStatusTable() dynamo.Table {
	environment := appConfig.GetEnvironment()
	cfg, err := appConfig.LoadConfigForEnvironment(environment)
	if err != nil {
		log.Fatal(err)
	}
	db := dynamo.New(session.New(), &aws.Config{Region: aws.String(cfg.AWSRegion)})
	return db.Table("OfferStatus")
}

var testEndpoint = getTestEndpoint()

func getTestEndpoint() string {
	testEndpoint := flag.String("testEndpoint", "http://staging-rex.us-west-2.elasticbeanstalk.com/", "The endpoint against which to run the integrations tests")
	flag.Parse()
	return *testEndpoint
}

func TestOfferStatuses(t *testing.T) {

	client := testConfig.GetRexClient(testEndpoint)

	const numOffers = 6
	var offerIDs [numOffers]string
	for i := 0; i < numOffers; i++ {
		offerIDs[i] = fmt.Sprintf("integTest%d", i)
	}

	Convey("Testing Get and Update OfferStatuses APIs", t, func() {
		Convey("setting and then getting and then setting offer statuses", func() {

			// Setting Statuses and validating them
			statuses := make(map[string]rex.OfferStatus)
			statuses[offerIDs[0]] = rex.OfferStatus_UNSEEN
			statuses[offerIDs[1]] = rex.OfferStatus_UNSEEN
			req := &rex.UpdateOfferStatusesRequest{
				TwitchUserID:  qaSamusTurbo,
				OfferStatuses: statuses,
			}

			resp, err := client.UpdateOfferStatuses(context.TODO(), req)
			So(err, ShouldBeNil)
			So(resp.OfferStatuses, ShouldResemble, statuses)

			err = validateStatuses(client, statuses)
			So(err, ShouldBeNil)

			// Changing Statuses and validating them
			newStatuses := make(map[string]rex.OfferStatus)
			newStatuses[offerIDs[0]] = rex.OfferStatus_SEEN
			newStatuses[offerIDs[1]] = rex.OfferStatus_SEEN
			req = &rex.UpdateOfferStatusesRequest{
				TwitchUserID:  qaSamusTurbo,
				OfferStatuses: newStatuses,
			}

			resp, err = client.UpdateOfferStatuses(context.TODO(), req)
			So(err, ShouldBeNil)
			So(resp.OfferStatuses, ShouldResemble, newStatuses)

			err = validateStatuses(client, newStatuses)
			So(err, ShouldBeNil)

			// Updating with no status and validating no changes
			emptyStatuses := make(map[string]rex.OfferStatus)
			emptyStatuses[offerIDs[0]] = rex.OfferStatus_NONE
			emptyStatuses[offerIDs[1]] = rex.OfferStatus_NONE
			req = &rex.UpdateOfferStatusesRequest{
				TwitchUserID:  qaSamusTurbo,
				OfferStatuses: emptyStatuses,
			}

			resp, err = client.UpdateOfferStatuses(context.TODO(), req)
			So(err, ShouldBeNil)
			So(resp, ShouldResemble, &rex.UpdateOfferStatusesResponse{OfferStatuses: nil})

			err = validateStatuses(client, newStatuses)
			So(err, ShouldBeNil)

		})

		Convey("Updating offer statuses to SEEN ignores any offers in a state other than UNSEEN", func() {

			// Setting and validating statuses for offers of every status
			originalStatuses := make(map[string]rex.OfferStatus)
			// Zero-th offer is a new offer and doesn't get set here
			originalStatuses[offerIDs[1]] = rex.OfferStatus_UNSEEN
			originalStatuses[offerIDs[2]] = rex.OfferStatus_SEEN
			originalStatuses[offerIDs[3]] = rex.OfferStatus_DISMISSED
			originalStatuses[offerIDs[4]] = rex.OfferStatus_OVERRIDDEN
			originalStatuses[offerIDs[5]] = rex.OfferStatus_CLAIMED
			req := &rex.UpdateOfferStatusesRequest{
				TwitchUserID:  qaSamusTurbo,
				OfferStatuses: originalStatuses,
			}

			resp, err := client.UpdateOfferStatuses(context.TODO(), req)
			So(err, ShouldBeNil)
			So(resp.OfferStatuses, ShouldResemble, originalStatuses)

			err = validateStatuses(client, originalStatuses)
			So(err, ShouldBeNil)

			// Updating statuses of all offers to SEEN
			updateStatusRequest := make(map[string]rex.OfferStatus)
			updateStatusRequest[offerIDs[0]] = rex.OfferStatus_SEEN
			updateStatusRequest[offerIDs[1]] = rex.OfferStatus_SEEN
			updateStatusRequest[offerIDs[2]] = rex.OfferStatus_SEEN
			updateStatusRequest[offerIDs[3]] = rex.OfferStatus_SEEN
			updateStatusRequest[offerIDs[4]] = rex.OfferStatus_SEEN
			updateStatusRequest[offerIDs[5]] = rex.OfferStatus_SEEN
			req = &rex.UpdateOfferStatusesRequest{
				TwitchUserID:  qaSamusTurbo,
				OfferStatuses: updateStatusRequest,
			}

			resp, err = client.UpdateOfferStatuses(context.TODO(), req)
			So(err, ShouldBeNil)

			// Validating that only the first two offers (the new offer and the UNSEEN offer) were changed to SEEN
			expectedResponse := make(map[string]rex.OfferStatus)
			expectedResponse[offerIDs[0]] = rex.OfferStatus_SEEN
			expectedResponse[offerIDs[1]] = rex.OfferStatus_SEEN
			So(resp.OfferStatuses, ShouldResemble, expectedResponse)

			originalStatuses[offerIDs[0]] = rex.OfferStatus_SEEN
			originalStatuses[offerIDs[1]] = rex.OfferStatus_SEEN
			err = validateStatuses(client, originalStatuses)
			So(err, ShouldBeNil)
		})

		// Clean up offers after each test
		Reset(func() {
			for _, offerID := range offerIDs {
				deleteOfferFromTable(qaSamusTurbo, offerID)
			}
		})
	})
}

func deleteOfferFromTable(twitchID string, offerID string) {
	offerStatusTable.Delete("twitchId", twitchID).Range("offerId", offerID).Run()
}

func validateStatuses(client rex.Rex, statuses map[string]rex.OfferStatus) error {
	offerIDs := []string{}
	for k := range statuses {
		offerIDs = append(offerIDs, k)
	}
	return retry(3, time.Second, func() error {
		req := &rex.GetOfferStatusesRequest{
			TwitchUserID: qaSamusTurbo,
			OfferIDs:     offerIDs,
		}
		resp, err := client.GetOfferStatuses(context.TODO(), req)
		if err != nil {
			// This error will result in a retry
			return err
		}
		for k, v := range statuses {
			if resp.OfferStatuses[k] != v {
				return errors.New(fmt.Sprintf("Status did not match. Expected %v, Actual %v", v, resp.OfferStatuses[k]))
			}
		}
		return nil
	})
}

func retry(attempts int, sleep time.Duration, fn func() error) error {
	if err := fn(); err != nil {

		if attempts--; attempts > 0 {
			time.Sleep(sleep)
			return retry(attempts, 2*sleep, fn)
		}
		return err
	}
	return nil
}
