package api

import (
	"net/http"

	log "github.com/sirupsen/logrus"

	"github.com/cactus/go-statsd-client/statsd"

	rrpc "code.justin.tv/samus/rex/rpc"

	"time"

	"code.justin.tv/foundation/twitchserver"
	"code.justin.tv/samus/rex/config"
	"code.justin.tv/samus/rex/metrics"
	"code.justin.tv/samus/rex/middleware"
	"goji.io"
	"goji.io/pat"
)

// RexServer API
type RexServer struct {
	OfferStatusAPI
	GiftingAPI
}

// Server struct
type Server struct {
	*goji.Mux
}

// NewServer defines handlers
func (n *RexServer) NewServer(stats statsd.Statter, cfg *config.Configuration) (*Server, error) {
	server := twitchserver.NewServer()

	twitchserver.AddDefaultSignalHandlers()
	s := &Server{
		server,
	}

	metricsConfig := metrics.Config{
		Stage:                          cfg.Environment,
		AwsRegion:                      cfg.AWSRegion,
		BufferSize:                     cfg.CloudwatchBufferSize,
		BatchSize:                      cfg.CloudwatchBatchSize,
		FlushInterval:                  time.Second * time.Duration(cfg.CloudwatchFlushIntervalSeconds),
		FlushPollCheckDelay:            time.Millisecond * time.Duration(cfg.CloudwatchFlushCheckDelayMs),
		BufferEmergencyFlushPercentage: cfg.CloudwatchEmergencyFlushPercentage,
	}

	metricLogger, metricsFlusher := metrics.New(metricsConfig)
	metricsMiddleware := middleware.NewMetricsMiddleware(metricLogger)
	go metricsFlusher.FlushMetricsAtInterval()

	// Add Metrics and Panic Recover Middleware
	s.Use(middleware.AccessLogger)
	s.Use(metricsMiddleware.Metrics)

	s.HandleFunc(pat.Get("/"), s.everythingIsFine)
	s.HandleFunc(pat.Get("/health"), s.everythingIsFine)
	s.HandleFunc(pat.Get("/debug/running"), s.everythingIsFine)

	twirpHandler := rrpc.NewRexServer(n, nil, nil)
	s.Handle(pat.Post(rrpc.RexPathPrefix+"*"), twirpHandler)

	return s, nil
}

func (s *Server) everythingIsFine(w http.ResponseWriter, r *http.Request) {
	_, err := w.Write([]byte("OK"))
	if err != nil {
		log.Fatal(err)
	}
}
