package api

import (
	"strconv"

	"golang.org/x/net/context"

	"code.justin.tv/samus/rex/internal/app/gifting"
	giftingClient "code.justin.tv/samus/rex/internal/clients/gifting"
	rrpc "code.justin.tv/samus/rex/rpc"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

type GiftingAPI struct {
	Gifting gifting.Gifting
}

func convertToProtobufGift(gift *giftingClient.Gift) (*rrpc.Gift, error) {
	if gift == nil {
		return nil, nil
	}

	protobuffStatus, err := convertToGiftStatus(gift.Status)

	return &rrpc.Gift{
		GiftID:  gift.GiftID,
		From:    gift.From,
		OfferID: gift.OfferID,
		To:      gift.To,
		Status:  protobuffStatus,
	}, err
}

func convertToGiftStatus(giftStatus string) (rrpc.GiftStatus, error) {
	protobuffStatus, ok := rrpc.GiftStatus_value[giftStatus]

	if !ok {
		return rrpc.GiftStatus_GIFT_CLAIM_STATUS_NONE, errors.New("Could not find protobuf status for " + giftStatus)
	} else {
		switch protobuffStatus {
		case int32(rrpc.GiftStatus_GIFT_CLAIM_PENDING):
			return rrpc.GiftStatus_GIFT_CLAIM_PENDING, nil
		case int32(rrpc.GiftStatus_GIFT_CLAIMED):
			return rrpc.GiftStatus_GIFT_CLAIMED, nil
		default:
			return rrpc.GiftStatus_GIFT_CLAIMED, errors.New("Could not find protobuf value for " + strconv.Itoa(int(protobuffStatus)) + " Gift status : " + giftStatus)
		}
	}
}

// GiftOffer gifts an offer to a user
func (g *GiftingAPI) GiftOffer(context context.Context, request *rrpc.GiftOfferRequest) (*rrpc.GiftOfferResponse, error) {
	logger := log.WithFields(log.Fields{
		"fromTwitchUserID": request.FromTwitchUserID,
		"offerID":          request.OfferID,
		"toTwitchUserID":   request.ToTwitchUserID,
	})

	logger.Info("[GiftOffer]")

	if request.FromTwitchUserID == "" {
		return nil, errors.New("FromTwitchUserID is a required field !")
	}

	if request.OfferID == "" {
		return nil, errors.New("OfferID is a required field !")
	}

	if request.ToTwitchUserID == "" {
		return nil, errors.New("ToTwitchUserID is a required field !")
	}

	gift, err := g.Gifting.CreateGift(context, request.FromTwitchUserID, request.OfferID, request.ToTwitchUserID)

	if err > rrpc.GiftOfferError_GIFT_OFFER_ERROR_NONE {
		logger.Error("[GiftOffer] Application responded with an error : ", err)
	}

	convertedGift, conversionErr := convertToProtobufGift(gift)

	if convertedGift == nil {
		logger.Warn("[GiftOffer] Converted gift was null.")
	}

	return &rrpc.GiftOfferResponse{
		Gift:  convertedGift,
		Error: err,
	}, conversionErr
}

func (g *GiftingAPI) GetClaimableGiftedOffer(context context.Context, request *rrpc.GetClaimableGiftedOfferRequest) (*rrpc.GetClaimableGiftedOfferResponse, error) {
	logger := log.WithFields(log.Fields{
		"twitchUserID": request.TwitchUserID,
		"offerID":      request.OfferID,
	})

	logger.Info("[GetClaimableGiftedOffer]")

	if request.TwitchUserID == "" {
		return nil, errors.New("TwitchUserID is a required field !")
	}

	if request.OfferID == "" {
		return nil, errors.New("OfferID is a required field !")
	}

	gift, err := g.Gifting.GetClaimableGift(context, request.TwitchUserID, request.OfferID)
	convertedGift, conversionErr := convertToProtobufGift(gift)

	if convertedGift == nil {
		logger.Warn("[GetClaimableGiftedOffer] Converted gift was null.")
	}

	if conversionErr != nil {
		if err != nil {
			err = errors.Wrap(err, conversionErr.Error())
		} else {
			err = conversionErr
		}
	}

	return &rrpc.GetClaimableGiftedOfferResponse{
		Gift: convertedGift,
	}, err
}

//GetGiftedOffersFrom returns the offers that have been gifted from a provided tuid.
func (g *GiftingAPI) GetGiftedOffersFrom(context context.Context, request *rrpc.GetGiftedOffersFromRequest) (*rrpc.GetGiftedOffersFromResponse, error) {
	logger := log.WithFields(log.Fields{
		"twitchUserID": request.TwitchUserID,
	})

	logger.Info("[GetGiftedOffersFrom]")

	if request.TwitchUserID == "" {
		return nil, errors.New("TwitchUserID is a required field !")
	}

	gifts, err := g.Gifting.GetGiftsFrom(context, request.TwitchUserID, request.OfferID)
	dGifts := *gifts

	protoGifts := make([]*rrpc.Gift, 0)
	for i := range dGifts {
		convertedGift, conversionErr := convertToProtobufGift(&dGifts[i])

		if conversionErr != nil {
			if err != nil {
				err = errors.Wrap(err, conversionErr.Error())
			} else {
				err = conversionErr
			}
		}

		protoGifts = append(protoGifts, convertedGift)
	}

	return &rrpc.GetGiftedOffersFromResponse{
		Gifts: protoGifts,
	}, err
}

//GetGiftedOffersTo returns the offers that have been gifted to a provided tuid.
func (g *GiftingAPI) GetGiftedOffersTo(context context.Context, request *rrpc.GetGiftedOffersToRequest) (*rrpc.GetGiftedOffersToResponse, error) {
	logger := log.WithFields(log.Fields{
		"twitchUserID": request.TwitchUserID,
	})

	logger.Info("[GetGiftedOffersTo]")

	if request.TwitchUserID == "" {
		return nil, errors.New("TwitchUserID is a required field !")
	}

	gifts, err := g.Gifting.GetGiftsTo(context, request.TwitchUserID, request.OfferID)
	dGifts := *gifts

	protoGifts := make([]*rrpc.Gift, 0)
	for i := range dGifts {

		convertedGift, conversionErr := convertToProtobufGift(&dGifts[i])

		if conversionErr != nil {
			if err != nil {
				err = errors.Wrap(err, conversionErr.Error())
			} else {
				err = conversionErr
			}
		}

		if convertedGift != nil {
			protoGifts = append(protoGifts, convertedGift)
		}
	}

	return &rrpc.GetGiftedOffersToResponse{
		Gifts: protoGifts,
	}, err
}

//ClaimGiftedOffer marks a gift as claimed.
func (g *GiftingAPI) ClaimGiftedOffer(context context.Context, request *rrpc.ClaimGiftedOfferRequest) (*rrpc.ClaimGiftedOfferResponse, error) {
	logger := log.WithFields(log.Fields{
		"giftID":       request.GiftID,
		"twitchUserID": request.TwitchUserID,
	})

	logger.Info("[ClaimGiftedOffer]")

	if request.GiftID == "" {
		return nil, errors.New("GiftID is a required field !")
	}

	claimedGift, err := g.Gifting.ClaimGift(context, request.GiftID, request.TwitchUserID)

	if err > rrpc.ClaimGiftedOfferError_GIFT_CLAIM_ERROR_NONE {
		logger.Error("[ClaimGiftedOffer] Application responded with an error : ", err)
	}

	convertedGift, conversionErr := convertToProtobufGift(claimedGift)

	if convertedGift == nil {
		logger.Warn("[ClaimGiftedOffer] Converted gift was null for giftID : ", request.GiftID)
	}

	return &rrpc.ClaimGiftedOfferResponse{
		ClaimedGift: convertedGift,
		Error:       err,
	}, conversionErr
}
