package api

import (
	"errors"
	"testing"

	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"

	giftingClient "code.justin.tv/samus/rex/internal/clients/gifting"
	mocks "code.justin.tv/samus/rex/mocks"
	"code.justin.tv/samus/rex/rpc"
)

func createPendingGift(giftID string, fromTUID string, offerID string, toTUID string) giftingClient.Gift {
	return giftingClient.Gift{
		GiftID:  giftID,
		From:    fromTUID,
		OfferID: offerID,
		To:      toTUID,
		Status:  "GIFT_CLAIM_PENDING",
	}
}

func TestGiftOffer(t *testing.T) {
	Convey("TestGiftOffer", t, func() {
		app := new(mocks.Gifting)
		api := GiftingAPI{
			Gifting: app,
		}

		Convey("Test GiftOffer", func() {
			Convey("When missing Gifter TUID", func() {

				req := &rex.GiftOfferRequest{
					FromTwitchUserID: "",
					OfferID:          "1",
					ToTwitchUserID:   "456",
				}
				_, err := api.GiftOffer(nil, req)
				So(err, ShouldNotBeNil)
			})

			Convey("When missing Offer ID", func() {

				req := &rex.GiftOfferRequest{
					FromTwitchUserID: "123",
					OfferID:          "",
					ToTwitchUserID:   "456",
				}
				_, err := api.GiftOffer(nil, req)
				So(err, ShouldNotBeNil)
			})

			Convey("When missing Giftee TUID", func() {

				req := &rex.GiftOfferRequest{
					FromTwitchUserID: "123",
					OfferID:          "1",
					ToTwitchUserID:   "",
				}
				_, err := api.GiftOffer(nil, req)
				So(err, ShouldNotBeNil)
			})

			Convey("When called with adequate input", func() {
				giftID := "gift-i-d"
				gifterTUID := "123"
				giftedOfferID := "1"
				gifteeTUID := "456"

				gift := createPendingGift(giftID, gifterTUID, giftedOfferID, gifteeTUID)
				convertedGift, conversionErr := convertToProtobufGift(&gift)

				So(conversionErr, ShouldBeNil)

				req := &rex.GiftOfferRequest{
					FromTwitchUserID: gifterTUID,
					OfferID:          giftedOfferID,
					ToTwitchUserID:   gifteeTUID,
				}

				app.On("CreateGift", mock.Anything, gifterTUID, giftedOfferID, gifteeTUID).Return(&gift, rex.GiftOfferError_GIFT_OFFER_ERROR_NONE)

				result, err := api.GiftOffer(nil, req)
				So(err, ShouldBeNil)
				So(result, ShouldResemble, &rex.GiftOfferResponse{Gift: convertedGift, Error: rex.GiftOfferError_GIFT_OFFER_ERROR_NONE})
			})

			Convey("When an application error occurs", func() {
				giftID := "gift-i-d"
				gifterTUID := "123"
				giftedOfferID := "1"
				gifteeTUID := "456"

				gift := createPendingGift(giftID, gifterTUID, giftedOfferID, gifteeTUID)
				convertedGift, conversionErr := convertToProtobufGift(&gift)

				So(conversionErr, ShouldBeNil)

				req := &rex.GiftOfferRequest{
					FromTwitchUserID: gifterTUID,
					OfferID:          giftedOfferID,
					ToTwitchUserID:   gifteeTUID,
				}

				app.On("CreateGift", mock.Anything, gifterTUID, giftedOfferID, gifteeTUID).Return(&gift, rex.GiftOfferError_GIFT_OFFER_ERROR_UNKNOWN_ERROR)

				result, err := api.GiftOffer(nil, req)
				So(err, ShouldBeNil)
				So(result, ShouldResemble, &rex.GiftOfferResponse{Gift: convertedGift, Error: rex.GiftOfferError_GIFT_OFFER_ERROR_UNKNOWN_ERROR})
			})
		})
	})
}

func TestGetClaimableGiftedOffer(t *testing.T) {
	Convey("TestGetClaimableGiftedOffer", t, func() {
		app := new(mocks.Gifting)
		api := GiftingAPI{
			Gifting: app,
		}

		userID := "a-t-u-i-d"
		offerID := "offer this !"

		Convey("Test GetClaimableGiftedOffer", func() {
			Convey("When a TUID is not provided", func() {
				req := &rex.GetClaimableGiftedOfferRequest{
					TwitchUserID: "",
					OfferID:      offerID,
				}

				response, err := api.GetClaimableGiftedOffer(nil, req)
				So(err, ShouldNotBeNil)
				So(response, ShouldBeNil)
			})

			Convey("When a OfferID is not provided", func() {
				req := &rex.GetClaimableGiftedOfferRequest{
					TwitchUserID: userID,
				}

				response, err := api.GetClaimableGiftedOffer(nil, req)
				So(err, ShouldNotBeNil)
				So(response, ShouldBeNil)
			})

			Convey("When proper parameters are provided", func() {
				fromUser := "123"
				giftID := "abcde"

				gift := createPendingGift(giftID, fromUser, offerID, userID)

				req := &rex.GetClaimableGiftedOfferRequest{
					TwitchUserID: userID,
					OfferID:      offerID,
				}

				Convey("When the application returns an error", func() {
					app.On("GetClaimableGift", mock.Anything, userID, offerID).Return(&gift, errors.New("Error of minute importance."))

					response, err := api.GetClaimableGiftedOffer(nil, req)
					So(err, ShouldNotBeNil)
					So(response, ShouldNotBeNil)
					So(response.Gift, ShouldNotBeNil) // Note the response will have a gift if the application returns one, even if there is an error.
				})

				Convey("When the application returns successfully", func() {
					app.On("GetClaimableGift", mock.Anything, userID, offerID).Return(&gift, nil)

					response, err := api.GetClaimableGiftedOffer(nil, req)
					So(err, ShouldBeNil)
					So(response, ShouldNotBeNil)
					So(response.Gift, ShouldNotBeNil)
				})
			})
		})
	})
}

func TestGetGiftedOffersFrom(t *testing.T) {
	Convey("TestGetGiftedOffersFrom", t, func() {
		app := new(mocks.Gifting)
		api := GiftingAPI{
			Gifting: app,
		}

		Convey("Test GetGiftedOffersFrom", func() {
			Convey("When missing TUID", func() {

				req := &rex.GetGiftedOffersFromRequest{
					TwitchUserID: "",
					OfferID:      "1",
				}
				_, err := api.GetGiftedOffersFrom(nil, req)
				So(err, ShouldNotBeNil)
			})

			Convey("When missing Offer ID", func() {
				// Calling w/o an offerID is an acceptable use case.
				tuid := "123"
				tuid2 := "456"
				tuid3 := "789"
				offerID := ""

				giftID1 := "abcde"
				giftID2 := "fghk"

				gift1 := createPendingGift(giftID1, tuid, "offer-123", tuid2)
				gift2 := createPendingGift(giftID2, tuid, "offer-456", tuid3)

				gifts := make([]giftingClient.Gift, 0)
				gifts = append(gifts, gift1)
				gifts = append(gifts, gift2)

				req := &rex.GetGiftedOffersFromRequest{
					TwitchUserID: tuid,
					OfferID:      offerID,
				}

				app.On("GetGiftsFrom", mock.Anything, tuid, offerID).Return(&gifts, nil)

				result, err := api.GetGiftedOffersFrom(nil, req)
				So(err, ShouldBeNil)
				So(len(result.Gifts), ShouldResemble, 2)
				So(result.Gifts[0].GiftID, ShouldResemble, giftID1)
				So(result.Gifts[1].GiftID, ShouldResemble, giftID2)
			})

			Convey("When including an Offer ID in the request", func() {
				tuid := "123"
				tuid2 := "456"
				tuid3 := "789"
				offerID := "the-offer-id"

				giftID1 := "abcde"
				giftID2 := "fghk"

				gift1 := createPendingGift(giftID1, tuid, offerID, tuid2)
				gift2 := createPendingGift(giftID2, tuid, offerID, tuid3)

				gifts := make([]giftingClient.Gift, 0)
				gifts = append(gifts, gift1)
				gifts = append(gifts, gift2)

				req := &rex.GetGiftedOffersFromRequest{
					TwitchUserID: tuid,
					OfferID:      offerID,
				}

				app.On("GetGiftsFrom", mock.Anything, tuid, offerID).Return(&gifts, nil)

				result, err := api.GetGiftedOffersFrom(nil, req)
				So(err, ShouldBeNil)
				So(len(result.Gifts), ShouldResemble, 2)
				So(result.Gifts[0].GiftID, ShouldResemble, giftID1)
				So(result.Gifts[1].GiftID, ShouldResemble, giftID2)
			})

			Convey("When an application error occurs", func() {
				tuid := "123"
				tuid2 := "456"
				tuid3 := "789"
				offerID := "the-offer-id"

				giftID1 := "abcde"
				giftID2 := "fghk"

				gift1 := createPendingGift(giftID1, tuid, offerID, tuid2)
				gift2 := createPendingGift(giftID2, tuid, offerID, tuid3)

				gifts := make([]giftingClient.Gift, 0)
				gifts = append(gifts, gift1)
				gifts = append(gifts, gift2)

				req := &rex.GetGiftedOffersFromRequest{
					TwitchUserID: tuid,
					OfferID:      offerID,
				}

				app.On("GetGiftsFrom", mock.Anything, tuid, offerID).Return(&gifts, errors.New("Test Error !"))

				_, err := api.GetGiftedOffersFrom(nil, req)
				So(err, ShouldNotBeNil)
			})

			Convey("When a data conversion error occurs", func() {
				tuid := "123"
				tuid2 := "456"
				tuid3 := "789"
				offerID := "the-offer-id"

				giftID1 := "abcde"
				giftID2 := "fghk"

				gift1 := createPendingGift(giftID1, tuid, offerID, tuid2)
				gift2 := createPendingGift(giftID2, tuid, offerID, tuid3)

				// Setting this to something unexpected will cause parsing to fail.
				gift1.Status = "JUNK"

				gifts := make([]giftingClient.Gift, 0)
				gifts = append(gifts, gift1)
				gifts = append(gifts, gift2)

				req := &rex.GetGiftedOffersFromRequest{
					TwitchUserID: tuid,
					OfferID:      offerID,
				}

				app.On("GetGiftsFrom", mock.Anything, tuid, offerID).Return(&gifts, nil)

				_, err := api.GetGiftedOffersFrom(nil, req)
				So(err, ShouldNotBeNil)
			})
		})
	})
}

func TestGetGiftedOffersTo(t *testing.T) {
	Convey("TestGetGiftedOffersTo", t, func() {
		app := new(mocks.Gifting)
		api := GiftingAPI{
			Gifting: app,
		}

		Convey("Test GetGiftedOffersTo", func() {
			Convey("When missing TUID", func() {

				req := &rex.GetGiftedOffersToRequest{
					TwitchUserID: "",
					OfferID:      "1",
				}
				_, err := api.GetGiftedOffersTo(nil, req)
				So(err, ShouldNotBeNil)
			})

			Convey("When missing Offer ID", func() {
				// Calling w/o an offerID is an acceptable use case.
				tuid := "123"
				tuid2 := "456"
				tuid3 := "789"
				offerID := ""

				giftID1 := "abcde"
				giftID2 := "fghk"

				gift1 := createPendingGift(giftID1, tuid2, "offer-123", tuid)
				gift2 := createPendingGift(giftID2, tuid3, "offer-456", tuid)

				gifts := make([]giftingClient.Gift, 0)
				gifts = append(gifts, gift1)
				gifts = append(gifts, gift2)

				req := &rex.GetGiftedOffersToRequest{
					TwitchUserID: tuid,
					OfferID:      offerID,
				}

				app.On("GetGiftsTo", mock.Anything, tuid, offerID).Return(&gifts, nil)

				result, err := api.GetGiftedOffersTo(nil, req)
				So(err, ShouldBeNil)
				So(len(result.Gifts), ShouldResemble, 2)
				So(result.Gifts[0].GiftID, ShouldResemble, giftID1)
				So(result.Gifts[1].GiftID, ShouldResemble, giftID2)
			})

			Convey("When including an Offer ID in the request", func() {
				tuid := "123"
				tuid2 := "456"
				tuid3 := "789"
				offerID := "the-offer-id"

				giftID1 := "abcde"
				giftID2 := "fghk"

				gift1 := createPendingGift(giftID1, tuid2, offerID, tuid)
				gift2 := createPendingGift(giftID2, tuid3, offerID, tuid)

				gifts := make([]giftingClient.Gift, 0)
				gifts = append(gifts, gift1)
				gifts = append(gifts, gift2)

				req := &rex.GetGiftedOffersToRequest{
					TwitchUserID: tuid,
					OfferID:      offerID,
				}

				app.On("GetGiftsTo", mock.Anything, tuid, offerID).Return(&gifts, nil)

				result, err := api.GetGiftedOffersTo(nil, req)
				So(err, ShouldBeNil)
				So(len(result.Gifts), ShouldResemble, 2)
				So(result.Gifts[0].GiftID, ShouldResemble, giftID1)
				So(result.Gifts[1].GiftID, ShouldResemble, giftID2)
			})

			Convey("When an application error occurs", func() {
				tuid := "123"
				tuid2 := "456"
				tuid3 := "789"
				offerID := "the-offer-id"

				giftID1 := "abcde"
				giftID2 := "fghk"

				gift1 := createPendingGift(giftID1, tuid, offerID, tuid2)
				gift2 := createPendingGift(giftID2, tuid, offerID, tuid3)

				gifts := make([]giftingClient.Gift, 0)
				gifts = append(gifts, gift1)
				gifts = append(gifts, gift2)

				req := &rex.GetGiftedOffersToRequest{
					TwitchUserID: tuid,
					OfferID:      offerID,
				}

				app.On("GetGiftsTo", mock.Anything, tuid, offerID).Return(&gifts, errors.New("Test Error !"))

				_, err := api.GetGiftedOffersTo(nil, req)
				So(err, ShouldNotBeNil)
			})

			Convey("When a data conversion error occurs", func() {
				tuid := "123"
				tuid2 := "456"
				tuid3 := "789"
				offerID := "the-offer-id"

				giftID1 := "abcde"
				giftID2 := "fghk"

				gift1 := createPendingGift(giftID1, tuid2, offerID, tuid)
				gift2 := createPendingGift(giftID2, tuid3, offerID, tuid)

				// Setting this to something unexpected will cause parsing to fail.
				gift1.Status = "JUNK"

				gifts := make([]giftingClient.Gift, 0)
				gifts = append(gifts, gift1)
				gifts = append(gifts, gift2)

				req := &rex.GetGiftedOffersToRequest{
					TwitchUserID: tuid,
					OfferID:      offerID,
				}

				app.On("GetGiftsTo", mock.Anything, tuid, offerID).Return(&gifts, nil)

				_, err := api.GetGiftedOffersTo(nil, req)
				So(err, ShouldNotBeNil)
			})
		})
	})
}

func TestClaimGiftedOffer(t *testing.T) {
	Convey("TestClaimGiftedOffer", t, func() {
		app := new(mocks.Gifting)
		api := GiftingAPI{
			Gifting: app,
		}

		Convey("Test ClaimGiftedOffer", func() {
			Convey("When missing Gift ID", func() {

				req := &rex.ClaimGiftedOfferRequest{
					GiftID: "",
				}
				_, err := api.ClaimGiftedOffer(nil, req)
				So(err, ShouldNotBeNil)
			})

			Convey("When the application does not return an error", func() {
				giftID := "a-gift-id"
				userID := "us-er-i-d"
				giftedOfferID := "1"
				gifteeTUID := "456"

				req := &rex.ClaimGiftedOfferRequest{
					GiftID:       giftID,
					TwitchUserID: userID,
				}

				gift := createPendingGift(giftID, userID, giftedOfferID, gifteeTUID)
				convertedGift, conversionErr := convertToProtobufGift(&gift)

				So(conversionErr, ShouldBeNil)

				app.On("ClaimGift", mock.Anything, giftID, userID).Return(&gift, rex.ClaimGiftedOfferError_GIFT_CLAIM_ERROR_NONE)

				result, err := api.ClaimGiftedOffer(nil, req)
				So(err, ShouldBeNil)
				So(result, ShouldNotBeNil)
				So(result.ClaimedGift, ShouldResemble, convertedGift)
				So(result.Error, ShouldEqual, rex.ClaimGiftedOfferError_GIFT_CLAIM_ERROR_NONE)
			})

			Convey("When the application returns an error.", func() {
				giftID := "a-gift-id"
				userID := "us-er-i-d"

				req := &rex.ClaimGiftedOfferRequest{
					GiftID:       giftID,
					TwitchUserID: userID,
				}

				app.On("ClaimGift", mock.Anything, giftID, userID).Return(nil, rex.ClaimGiftedOfferError_GIFT_CLAIM_ERROR_UNKNOWN_ERROR)

				result, err := api.ClaimGiftedOffer(nil, req)
				So(err, ShouldBeNil)
				So(result, ShouldNotBeNil)
				So(result.ClaimedGift, ShouldEqual, nil)
				So(result.Error, ShouldEqual, rex.ClaimGiftedOfferError_GIFT_CLAIM_ERROR_UNKNOWN_ERROR)
			})
		})
	})
}
