package api

import (
	"golang.org/x/net/context"

	"code.justin.tv/samus/rex/internal/app/status"
	rrpc "code.justin.tv/samus/rex/rpc"
	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
)

type OfferStatusAPI struct {
	OfferStatus status.OfferStatus
}

// GetOfferStatuses gets a users offerStatuses for given TwitchUserID and OfferIDs
func (os *OfferStatusAPI) GetOfferStatuses(context context.Context, request *rrpc.GetOfferStatusesRequest) (*rrpc.GetOfferStatusesResponse, error) {
	logger := log.WithFields(log.Fields{
		"userID":   request.TwitchUserID,
		"offerIDs": request.OfferIDs,
	})

	logger.Info("[GetOfferStatuses]")

	if request.TwitchUserID == "" {
		return nil, errors.New("TwitchUserID is a required field!")
	}
	if len(request.OfferIDs) == 0 {
		return nil, errors.New("OfferIDs is a required field!")
	}

	statuses, err := os.OfferStatus.GetOfferStatuses(context, request.TwitchUserID, request.OfferIDs)
	if err != nil {
		logger.Error("[GetOfferStatuses] : ", err)
		return nil, errors.Wrap(err, "Error calling GetOfferStatuses")
	}

	return &rrpc.GetOfferStatusesResponse{
		OfferStatuses: statuses,
	}, nil
}

// UpdateOfferStatuses updates a user's Offer Statuses
func (os *OfferStatusAPI) UpdateOfferStatuses(context context.Context, request *rrpc.UpdateOfferStatusesRequest) (*rrpc.UpdateOfferStatusesResponse, error) {
	logger := log.WithFields(log.Fields{
		"userID":        request.TwitchUserID,
		"offerStatuses": request.OfferStatuses,
	})

	logger.Info("[UpdateOfferStatuses]")

	if request.TwitchUserID == "" {
		return nil, errors.New("TwitchUserID is a required field!")
	}
	if len(request.OfferStatuses) == 0 {
		return nil, errors.New("OfferStatuses is a required field!")
	}

	filteredOfferStatuses, err := os.filterOfferStatuses(context, request.TwitchUserID, request.OfferStatuses)
	if err != nil {
		logger.Error("[UpdateOfferStatuses] error filtering offer statuses : ", err)
		return nil, errors.Wrap(err, "Error filtering offer statuses when calling UpdateOfferStatuses")
	}
	if len(filteredOfferStatuses) == 0 {
		logger.Warn("[UpdateOfferStatuses] : No offers with valid status!")
		return &rrpc.UpdateOfferStatusesResponse{
			OfferStatuses: nil,
		}, nil
	}

	err = os.OfferStatus.UpdateOfferStatuses(context, request.TwitchUserID, filteredOfferStatuses)
	if err != nil {
		logger.Error("[UpdateOfferStatuses] : ", err)
		return nil, errors.Wrap(err, "Error calling UpdateOfferStatuses")
	}

	return &rrpc.UpdateOfferStatusesResponse{
		OfferStatuses: filteredOfferStatuses,
	}, nil
}

/** Removes empty updates and invalid offer updates.
 * Invalid offer updates are any updates changing an offer's state to SEEN if the offer already existed in a state other than UNSEEN.
 * We discard those because offers should only ever become seen if the user hasn't seen/interacted with them before. We don't want
 * to mark an offer that's already in a SEEN/DISMISSED/CLAIMED/OVERRIDDEN state as SEEN. */
func (os *OfferStatusAPI) filterOfferStatuses(context context.Context, userID string, statuses map[string]rrpc.OfferStatus) (map[string]rrpc.OfferStatus, error) {
	updatesToNoneFilteredOut := make(map[string]rrpc.OfferStatus)
	for offerID, status := range statuses {
		if status != rrpc.OfferStatus_NONE {
			updatesToNoneFilteredOut[offerID] = status
		} else {
			logger := log.WithFields(log.Fields{
				"offerID": offerID,
			})
			logger.Warn("[UpdateOfferStatuses] : filtered out offer with empty status")
		}
	}

	var offerIDsToMarkAsSeen []string
	invalidUpdatesToSeenFilteredOut := make(map[string]rrpc.OfferStatus)

	// Extract offers we want to mark as SEEN
	for offerID, status := range updatesToNoneFilteredOut {
		if status == rrpc.OfferStatus_SEEN {
			offerIDsToMarkAsSeen = append(offerIDsToMarkAsSeen, offerID)
		} else {
			invalidUpdatesToSeenFilteredOut[offerID] = status
		}
	}

	// Filter out any offer updates to SEEN if the offer already existed in a state other than UNSEEN
	if len(offerIDsToMarkAsSeen) > 0 {
		discardedSeenUpdates := make(map[string]rrpc.OfferStatus)
		currentStatusesOfOffersToMarkAsSeen, err := os.OfferStatus.GetOfferStatuses(context, userID, offerIDsToMarkAsSeen)
		if err != nil {
			logger := log.WithFields(log.Fields{
				"userID":               userID,
				"offerIDsToMarkAsSeen": offerIDsToMarkAsSeen,
			})
			logger.Error("[UpdateOfferStatuses] failed to read the current state of some offers to be marked as seen : ", err)
			return nil, err
		} else {
			for offerID, status := range currentStatusesOfOffersToMarkAsSeen {
				if status == rrpc.OfferStatus_UNSEEN {
					invalidUpdatesToSeenFilteredOut[offerID] = rrpc.OfferStatus_SEEN
				} else {
					discardedSeenUpdates[offerID] = rrpc.OfferStatus_SEEN
				}
			}
			if len(discardedSeenUpdates) > 0 {
				logger := log.WithFields(log.Fields{
					"userID":               userID,
					"discardedSeenUpdates": discardedSeenUpdates,
				})
				logger.Warn("[UpdateOfferStatuses] : tried to mark some offers as SEEN, but they already had a status other than UNSEEN")
			}
		}
	}

	return invalidUpdatesToSeenFilteredOut, nil
}
