package api

import (
	"testing"

	mocks "code.justin.tv/samus/rex/mocks"
	rex "code.justin.tv/samus/rex/rpc"
	"github.com/pkg/errors"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func TestGetOfferStatuses(t *testing.T) {
	Convey("TestGetOfferStatuses", t, func() {
		app := new(mocks.OfferStatus)
		api := OfferStatusAPI{
			OfferStatus: app,
		}

		Convey("Test GetOfferStatus", func() {
			Convey("when missing twitch user id", func() {

				req := &rex.GetOfferStatusesRequest{
					TwitchUserID: "",
					OfferIDs:     []string{"1"},
				}
				_, err := api.GetOfferStatuses(nil, req)
				So(err, ShouldNotBeNil)
			})

			Convey("when missing offer statuses", func() {

				req := &rex.GetOfferStatusesRequest{
					TwitchUserID: "123456",
					OfferIDs:     []string{},
				}
				_, err := api.GetOfferStatuses(nil, req)
				So(err, ShouldNotBeNil)
			})

			Convey("When called with valid input", func() {
				statuses := make(map[string]rex.OfferStatus)
				statuses["1"] = rex.OfferStatus_SEEN
				offerIDs := []string{"1"}

				req := &rex.GetOfferStatusesRequest{
					TwitchUserID: "123456",
					OfferIDs:     offerIDs,
				}
				app.On("GetOfferStatuses", mock.Anything, mock.Anything, mock.Anything).Return(statuses, nil)

				result, err := api.GetOfferStatuses(nil, req)
				So(err, ShouldBeNil)
				So(result, ShouldResemble, &rex.GetOfferStatusesResponse{OfferStatuses: statuses})
			})

			Convey("When an error happens", func() {
				statuses := make(map[string]rex.OfferStatus)
				offerIDs := []string{"1"}

				req := &rex.GetOfferStatusesRequest{
					TwitchUserID: "123456",
					OfferIDs:     offerIDs,
				}
				app.On("GetOfferStatuses", mock.Anything, mock.Anything, mock.Anything).Return(statuses, errors.New("error"))

				_, err := api.GetOfferStatuses(nil, req)
				So(err, ShouldNotBeNil)
			})
		})
	})
}

func TestUpdateOfferStatuses(t *testing.T) {
	Convey("TestUpdateOfferStatuses", t, func() {
		app := new(mocks.OfferStatus)
		api := OfferStatusAPI{
			OfferStatus: app,
		}

		Convey("Test UpdateOfferStatuses", func() {
			Convey("when missing twitch user id", func() {
				statuses := make(map[string]rex.OfferStatus)
				statuses["1"] = rex.OfferStatus_SEEN

				req := &rex.UpdateOfferStatusesRequest{
					TwitchUserID:  "",
					OfferStatuses: statuses,
				}
				_, err := api.UpdateOfferStatuses(nil, req)
				So(err, ShouldNotBeNil)
			})

			Convey("when missing offer statuses", func() {
				statuses := make(map[string]rex.OfferStatus)

				req := &rex.UpdateOfferStatusesRequest{
					TwitchUserID:  "123456",
					OfferStatuses: statuses,
				}
				_, err := api.UpdateOfferStatuses(nil, req)
				So(err, ShouldNotBeNil)
			})

			Convey("When called with valid input", func() {
				oldStatuses := make(map[string]rex.OfferStatus)
				oldStatuses["1"] = rex.OfferStatus_UNSEEN
				newStatuses := make(map[string]rex.OfferStatus)
				newStatuses["1"] = rex.OfferStatus_SEEN

				req := &rex.UpdateOfferStatusesRequest{
					TwitchUserID:  "123456",
					OfferStatuses: newStatuses,
				}
				app.On("UpdateOfferStatuses", mock.Anything, mock.Anything, mock.Anything).Return(nil)
				app.On("GetOfferStatuses", mock.Anything, mock.Anything, mock.Anything).Return(oldStatuses, nil)

				result, err := api.UpdateOfferStatuses(nil, req)
				So(err, ShouldBeNil)
				So(result, ShouldResemble, &rex.UpdateOfferStatusesResponse{OfferStatuses: newStatuses})
			})

			Convey("When called with filtered input (discarded update because the status was NONE)", func() {
				oldStatuses := make(map[string]rex.OfferStatus)
				oldStatuses["1"] = rex.OfferStatus_UNSEEN

				newStatuses := make(map[string]rex.OfferStatus)
				newStatuses["1"] = rex.OfferStatus_SEEN
				newStatuses["2"] = rex.OfferStatus_NONE

				filteredStatuses := make(map[string]rex.OfferStatus)
				filteredStatuses["1"] = rex.OfferStatus_SEEN

				req := &rex.UpdateOfferStatusesRequest{
					TwitchUserID:  "123456",
					OfferStatuses: newStatuses,
				}
				app.On("UpdateOfferStatuses", mock.Anything, mock.Anything, mock.Anything).Return(nil)
				app.On("GetOfferStatuses", mock.Anything, mock.Anything, mock.Anything).Return(oldStatuses, nil)

				result, err := api.UpdateOfferStatuses(nil, req)
				So(err, ShouldBeNil)
				So(result, ShouldResemble, &rex.UpdateOfferStatusesResponse{OfferStatuses: filteredStatuses})
			})

			Convey("When called with filtered input (discarded SEEN updates because the offers were in a state other than UNSEEN)", func() {
				oldStatuses := make(map[string]rex.OfferStatus)
				oldStatuses["1"] = rex.OfferStatus_UNSEEN
				oldStatuses["2"] = rex.OfferStatus_SEEN
				oldStatuses["3"] = rex.OfferStatus_DISMISSED
				oldStatuses["4"] = rex.OfferStatus_OVERRIDDEN
				oldStatuses["5"] = rex.OfferStatus_CLAIMED

				newStatuses := make(map[string]rex.OfferStatus)
				newStatuses["1"] = rex.OfferStatus_SEEN
				newStatuses["2"] = rex.OfferStatus_SEEN
				newStatuses["3"] = rex.OfferStatus_SEEN
				newStatuses["4"] = rex.OfferStatus_SEEN
				newStatuses["5"] = rex.OfferStatus_SEEN

				filteredStatuses := make(map[string]rex.OfferStatus)
				filteredStatuses["1"] = rex.OfferStatus_SEEN

				req := &rex.UpdateOfferStatusesRequest{
					TwitchUserID:  "123456",
					OfferStatuses: newStatuses,
				}
				app.On("UpdateOfferStatuses", mock.Anything, mock.Anything, mock.Anything).Return(nil)
				app.On("GetOfferStatuses", mock.Anything, mock.Anything, mock.Anything).Return(oldStatuses, nil)

				result, err := api.UpdateOfferStatuses(nil, req)
				So(err, ShouldBeNil)
				So(result, ShouldResemble, &rex.UpdateOfferStatusesResponse{OfferStatuses: filteredStatuses})
			})

			Convey("When called with completely filtered input", func() {
				oldStatuses := make(map[string]rex.OfferStatus)
				oldStatuses["1"] = rex.OfferStatus_UNSEEN
				oldStatuses["2"] = rex.OfferStatus_UNSEEN
				newStatuses := make(map[string]rex.OfferStatus)
				newStatuses["1"] = rex.OfferStatus_NONE
				newStatuses["2"] = rex.OfferStatus_NONE

				req := &rex.UpdateOfferStatusesRequest{
					TwitchUserID:  "123456",
					OfferStatuses: newStatuses,
				}

				app.On("GetOfferStatuses", mock.Anything, mock.Anything, mock.Anything).Return(oldStatuses, nil)

				result, err := api.UpdateOfferStatuses(nil, req)
				So(err, ShouldBeNil)
				So(result, ShouldResemble, &rex.UpdateOfferStatusesResponse{OfferStatuses: nil})
			})

			Convey("When an error happens", func() {
				oldStatuses := make(map[string]rex.OfferStatus)
				oldStatuses["1"] = rex.OfferStatus_UNSEEN
				statuses := make(map[string]rex.OfferStatus)
				statuses["1"] = rex.OfferStatus_SEEN

				req := &rex.UpdateOfferStatusesRequest{
					TwitchUserID:  "123456",
					OfferStatuses: statuses,
				}
				app.On("UpdateOfferStatuses", mock.Anything, mock.Anything, mock.Anything).Return(errors.New("error"))
				app.On("GetOfferStatuses", mock.Anything, mock.Anything, mock.Anything).Return(oldStatuses, nil)

				_, err := api.UpdateOfferStatuses(nil, req)
				So(err, ShouldNotBeNil)
			})
		})
	})
}
