package gifting

import (
	"fmt"

	"github.com/pkg/errors"
	log "github.com/sirupsen/logrus"
	"golang.org/x/net/context"

	"code.justin.tv/samus/rex/internal/clients"
	giftingClient "code.justin.tv/samus/rex/internal/clients/gifting"
	rrpc "code.justin.tv/samus/rex/rpc"
)

type Gifting interface {
	CreateGift(ctx context.Context, fromID string, offerID string, toID string) (*giftingClient.Gift, rrpc.GiftOfferError)
	GetClaimableGift(ctx context.Context, userID string, offerID string) (*giftingClient.Gift, error)
	GetGiftsFrom(ctx context.Context, userID string, offerID string) (*[]giftingClient.Gift, error)
	GetGiftsTo(ctx context.Context, userID string, offerID string) (*[]giftingClient.Gift, error)
	ClaimGift(ctx context.Context, giftID string, userID string) (*giftingClient.Gift, rrpc.ClaimGiftedOfferError)
}

type GiftingOfferWhitelist struct {
	allowedOffers []string
	active        bool
}

func newGiftingOfferWhitelist(whitelist []string, whitelistActive bool) GiftingOfferWhitelist {
	return GiftingOfferWhitelist{
		allowedOffers: whitelist,
		active:        whitelistActive,
	}
}

func (w GiftingOfferWhitelist) canOfferBeGifted(offerID string) bool {
	if !w.active {
		return true
	}

	for _, s := range w.allowedOffers {
		if s == offerID {
			return true
		}
	}

	return false
}

func (w GiftingOfferWhitelist) isActive() bool {
	return w.active
}

type GiftingImpl struct {
	Clients        clients.Clients
	offerWhitelist GiftingOfferWhitelist
}

func NewGifting(c clients.Clients, whitelist []string, whitelistActive bool) Gifting {
	return &GiftingImpl{
		Clients:        c,
		offerWhitelist: newGiftingOfferWhitelist(whitelist, whitelistActive),
	}
}

const (
	GatewayClaimPayloadFormat string = `{"metadata":{"toUser": %#v, "fromUser": %#v}}`

	MaxNumberOfGiftsPerOffer int = 1
)

// CreateGift creates a gift.
func (s *GiftingImpl) CreateGift(ctx context.Context, fromID string, offerID string, toID string) (*giftingClient.Gift, rrpc.GiftOfferError) {
	logger := log.WithFields(log.Fields{
		"fromTwitchUserID": fromID,
		"offerID":          offerID,
		"toTwitchUserID":   toID,
	})

	if !s.offerWhitelist.canOfferBeGifted(offerID) {
		logger.Error("[CreateGift] Offer not in whitelist")
		return nil, rrpc.GiftOfferError_GIFT_OFFER_ERROR_OFFER_CANT_BE_GIFTED
	}

	if toID == fromID {
		logger.Error("[CreateGift] Cannot gift offer to yourself")
		return nil, rrpc.GiftOfferError_GIFT_OFFER_ERROR_GIFT_CREATION_ERROR
	}

	isPrime, err := s.Clients.NitroClient.IsPrime(ctx, fromID)

	if err != nil {
		logger.Error(errors.Wrap(err, "[CreateGift] Error checking prime status"))
		return nil, rrpc.GiftOfferError_GIFT_OFFER_ERROR_GIFT_CREATION_ERROR
	}

	if !isPrime {
		logger.Error("[CreateGift] User giving gift is not prime")
		return nil, rrpc.GiftOfferError_GIFT_OFFER_ERROR_GIFT_CREATION_ERROR
	}

	giftsFrom, err := s.Clients.GiftingClient.GetGiftsFromForOffer(ctx, fromID, offerID)

	for _, gift := range *giftsFrom {
		if gift.To == toID {
			return nil, rrpc.GiftOfferError_GIFT_OFFER_ERROR_GIFT_CREATION_ERROR
		}
	}

	if err != nil {
		logger.Error(errors.Wrap(err, "[CreateGift] Error getting gifts from for offer"))
		return nil, rrpc.GiftOfferError_GIFT_OFFER_ERROR_GIFT_CREATION_ERROR
	}

	if giftsFrom != nil && len(*giftsFrom) >= MaxNumberOfGiftsPerOffer {
		return nil, rrpc.GiftOfferError_GIFT_OFFER_ERROR_MAX_GIFTS_FOR_OFFER_EXCEEDED
	}

	gift, err := s.Clients.GiftingClient.CreateGift(ctx, fromID, offerID, toID)

	if gift == nil || err != nil {
		return gift, rrpc.GiftOfferError_GIFT_OFFER_ERROR_GIFT_CREATION_ERROR
	}

	claimableGift, err := s.Clients.GiftingClient.GetOrCreateClaimableGift(ctx, gift)

	if claimableGift == nil || err != nil {
		return gift, rrpc.GiftOfferError_GIFT_OFFER_ERROR_CLAIMABLE_GIFT_CREATION_ERROR
	}

	if claimableGift.GiftID == gift.GiftID {
		// First gift created for the offer. Notify !
		notificationError := s.Clients.NotificationsClient.SendGiftCreationNotification(ctx, gift)
		if notificationError != nil {
			return gift, rrpc.GiftOfferError_GIFT_OFFER_ERROR_NOTIFICATION_FAILURE
		}
	}

	return gift, rrpc.GiftOfferError_GIFT_OFFER_ERROR_NONE
}

// GetClaimableGift returns the gift that is claimable for a given user and offer.
func (s *GiftingImpl) GetClaimableGift(ctx context.Context, userID string, offerID string) (*giftingClient.Gift, error) {
	claimableGift, err := s.Clients.GiftingClient.GetClaimableGift(ctx, userID, offerID)

	if claimableGift != nil {
		gift, giftErr := s.Clients.GiftingClient.GetGift(ctx, claimableGift.GiftID)

		if err != nil {
			if giftErr == nil {
				giftErr = err
			} else {
				giftErr = errors.Wrap(err, giftErr.Error())
			}
		}

		return gift, giftErr
	}

	return nil, err
}

// GetGiftsFrom returns the gifts gifted by a provided user.
func (s *GiftingImpl) GetGiftsFrom(ctx context.Context, userID string, offerID string) (*[]giftingClient.Gift, error) {
	if len(offerID) > 0 {
		return s.Clients.GiftingClient.GetGiftsFromForOffer(ctx, userID, offerID)
	} else {
		return s.Clients.GiftingClient.GetGiftsFrom(ctx, userID)
	}
}

// GetGiftsTo returns the gifts gifted to a provided user.
func (s *GiftingImpl) GetGiftsTo(ctx context.Context, userID string, offerID string) (*[]giftingClient.Gift, error) {
	if len(offerID) > 0 {
		return s.Clients.GiftingClient.GetGiftsToForOffer(ctx, userID, offerID)
	} else {
		return s.Clients.GiftingClient.GetGiftsTo(ctx, userID)
	}
}

// ClaimGift marks a gift as claimed.
func (s *GiftingImpl) ClaimGift(ctx context.Context, giftID string, userID string) (*giftingClient.Gift, rrpc.ClaimGiftedOfferError) {
	gift, err := s.Clients.GiftingClient.GetGift(ctx, giftID)

	// Does the gift exist ?
	if gift == nil {
		return nil, rrpc.ClaimGiftedOfferError_GIFT_CLAIM_ERROR_NON_EXISTENT_GIFT
	}

	// Is the appropriate user claiming it ?
	if gift.To != userID {
		return nil, rrpc.ClaimGiftedOfferError_GIFT_CLAIM_ERROR_USER_DOES_NOT_OWN_GIFT
	}

	// Is it claimable ?
	if !gift.IsClaimable() {
		return nil, rrpc.ClaimGiftedOfferError_GIFT_CLAIM_ERROR_GIFT_NOT_CLAIMABLE
	}

	claimableGift, err := s.Clients.GiftingClient.GetClaimableGift(ctx, gift.To, gift.OfferID)

	// Is it the gift that is claimable for this offer ?
	if err != nil || claimableGift == nil {
		return nil, rrpc.ClaimGiftedOfferError_GIFT_CLAIM_ERROR_UNKNOWN_ERROR
	}

	if claimableGift.GiftID != giftID || claimableGift.Tuid != gift.To {
		return nil, rrpc.ClaimGiftedOfferError_GIFT_CLAIM_ERROR_GIFT_NOT_CLAIMABLE
	}

	// Claim the gift through samus gateway.
	payload := []byte(fmt.Sprintf(GatewayClaimPayloadFormat, gift.To, gift.From))
	claimOfferResponse, claimOfferError, err := s.Clients.GatewayClient.ClaimPrimeOffer(ctx, gift.To, gift.OfferID, "en", payload, nil)

	if err != nil || claimOfferError != nil || claimOfferResponse == nil {
		return nil, rrpc.ClaimGiftedOfferError_GIFT_CLAIM_ERROR_UNKNOWN_ERROR
	}

	// Update that the gift was claimed.
	updatedGift, err := s.Clients.GiftingClient.ClaimGift(ctx, giftID)

	if err != nil {
		return updatedGift, rrpc.ClaimGiftedOfferError_GIFT_CLAIM_ERROR_UNKNOWN_ERROR
	}

	return updatedGift, rrpc.ClaimGiftedOfferError_GIFT_CLAIM_ERROR_NONE
}
