package gifting

import (
	"testing"

	"github.com/pkg/errors"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"

	"code.justin.tv/samus/gateway/client"
	"code.justin.tv/samus/rex/internal/clients"
	giftingClient "code.justin.tv/samus/rex/internal/clients/gifting"
	mocks "code.justin.tv/samus/rex/mocks"
	gateway_mocks "code.justin.tv/samus/rex/mocks/samus_gateway"
	rrpc "code.justin.tv/samus/rex/rpc"
)

func createPendingGift(giftID string, fromTUID string, offerID string, toTUID string) giftingClient.Gift {
	return giftingClient.Gift{
		GiftID:  giftID,
		From:    fromTUID,
		OfferID: offerID,
		To:      toTUID,
		Status:  "GIFT_CLAIM_PENDING",
	}
}

func createClaimableGift(tuid string, offerID string, giftID string) giftingClient.ClaimableGift {
	return giftingClient.ClaimableGift{
		Tuid:    tuid,
		OfferID: offerID,
		GiftID:  giftID,
	}
}

func TestOfferWhitelist(t *testing.T) {
	Convey("TestOfferWhitelist", t, func() {
		offerID1 := "anOf-fer-eye-dee"
		offerID2 := "anOf-fer-eye-dee2"
		badOfferID := "badBoys4Life"
		whitelist := []string{offerID1, offerID2}
		emptyWhitelist := []string{}
		offerWhitelistActive := newGiftingOfferWhitelist(whitelist, true)
		offerWhitelistInactive := newGiftingOfferWhitelist(whitelist, false)
		emptyOfferWhitelist := newGiftingOfferWhitelist(emptyWhitelist, true)

		Convey("Test Active Whitelist only allows whitelisted offers", func() {
			So(true, ShouldEqual, offerWhitelistActive.canOfferBeGifted(offerID1))
			So(true, ShouldEqual, offerWhitelistActive.canOfferBeGifted(offerID2))
			So(false, ShouldEqual, offerWhitelistActive.canOfferBeGifted(badOfferID))
		})

		Convey("Test Inactive Whitelist lets everything through", func() {
			So(true, ShouldEqual, offerWhitelistInactive.canOfferBeGifted(offerID1))
			So(true, ShouldEqual, offerWhitelistInactive.canOfferBeGifted(offerID2))
			So(true, ShouldEqual, offerWhitelistInactive.canOfferBeGifted(badOfferID))
		})

		Convey("Test Active Empty Whitelist blocks all offers", func() {
			So(false, ShouldEqual, emptyOfferWhitelist.canOfferBeGifted(offerID1))
			So(false, ShouldEqual, emptyOfferWhitelist.canOfferBeGifted(offerID2))
			So(false, ShouldEqual, emptyOfferWhitelist.canOfferBeGifted(badOfferID))
		})
	})
}

func TestCreateGift(t *testing.T) {
	Convey("TestCreateGift", t, func() {
		giftClient := new(mocks.IGiftingClient)
		nitroClient := new(mocks.INitroClient)
		notificationsClient := new(mocks.INotificationsClient)
		giftedOfferID := "anOf-fer-eye-dee"
		whitelist := []string{giftedOfferID}
		theClients := clients.Clients{
			GiftingClient:       giftClient,
			NotificationsClient: notificationsClient,
			NitroClient:         nitroClient,
		}

		app := GiftingImpl{
			Clients:        theClients,
			offerWhitelist: newGiftingOfferWhitelist(whitelist, true),
		}

		Convey("Test CreateGift", func() {
			giftID := "gift-i-d"
			gifterTUID := "123"
			gifteeTUID := "456"
			priorGifteeTUID := "789"

			gift := createPendingGift(giftID, gifterTUID, giftedOfferID, priorGifteeTUID)
			claimableGift := createClaimableGift(gifteeTUID, giftedOfferID, giftID)

			nitroClient.On("IsPrime", mock.Anything, gifterTUID).Return(true, nil)

			Convey("When trying to gift an offer that is not in the whitelist", func() {
				badOfferID := "badOID"
				result, err := app.CreateGift(nil, gifterTUID, badOfferID, gifteeTUID)
				So(err, ShouldEqual, rrpc.GiftOfferError_GIFT_OFFER_ERROR_OFFER_CANT_BE_GIFTED)
				So(result, ShouldBeNil)
			})

			Convey("When determining the number of gifts given by the user returns an error", func() {
				priorGifts := make([]giftingClient.Gift, 0)
				giftClient.On("GetGiftsFromForOffer", mock.Anything, gifterTUID, giftedOfferID).Return(&priorGifts, errors.New("An error of modest importance"))

				result, err := app.CreateGift(nil, gifterTUID, giftedOfferID, gifteeTUID)
				So(err, ShouldEqual, rrpc.GiftOfferError_GIFT_OFFER_ERROR_GIFT_CREATION_ERROR)
				So(result, ShouldBeNil)
			})

			Convey("When the user has exceeded the gifting limit for the offer", func() {
				priorGifts := make([]giftingClient.Gift, 0)
				// 11 prior gifts.
				for i := 1; i <= 11; i++ {
					priorGifts = append(priorGifts, gift)
				}

				giftClient.On("GetGiftsFromForOffer", mock.Anything, gifterTUID, giftedOfferID).Return(&priorGifts, nil)

				result, err := app.CreateGift(nil, gifterTUID, giftedOfferID, gifteeTUID)
				So(err, ShouldEqual, rrpc.GiftOfferError_GIFT_OFFER_ERROR_MAX_GIFTS_FOR_OFFER_EXCEEDED)
				So(result, ShouldBeNil)
			})

			Convey("When the user has already gifted that user", func() {
				priorGifts := make([]giftingClient.Gift, 0)

				priorGifts = append(priorGifts, createPendingGift(giftID, gifterTUID, giftedOfferID, gifteeTUID))

				giftClient.On("GetGiftsFromForOffer", mock.Anything, gifterTUID, giftedOfferID).Return(&priorGifts, nil)

				result, err := app.CreateGift(nil, gifterTUID, giftedOfferID, gifteeTUID)
				So(err, ShouldEqual, rrpc.GiftOfferError_GIFT_OFFER_ERROR_GIFT_CREATION_ERROR)
				So(result, ShouldBeNil)
			})

			Convey("When the user is gifting themself", func() {
				result, err := app.CreateGift(nil, gifterTUID, giftedOfferID, gifterTUID)
				So(err, ShouldEqual, rrpc.GiftOfferError_GIFT_OFFER_ERROR_GIFT_CREATION_ERROR)
				So(result, ShouldBeNil)
			})

			Convey("When the user has not exceeded the gifting limit for the offer", func() {
				priorGifts := make([]giftingClient.Gift, 0)
				giftClient.On("GetGiftsFromForOffer", mock.Anything, gifterTUID, giftedOfferID).Return(&priorGifts, nil)

				Convey("When the client creates the gift without error", func() {
					giftClient.On("CreateGift", mock.Anything, gifterTUID, giftedOfferID, gifteeTUID).Return(&gift, nil)

					Convey("And the client gets or creates the claimable gift without error", func() {

						Convey("And this is the first gift given for the offer.", func() {
							giftClient.On("GetOrCreateClaimableGift", mock.Anything, &gift).Return(&claimableGift, nil)

							Convey("And the notification fires without problems", func() {
								notificationsClient.On("SendGiftCreationNotification", mock.Anything, mock.Anything).Return(nil)

								result, err := app.CreateGift(nil, gifterTUID, giftedOfferID, gifteeTUID)
								So(err, ShouldEqual, rrpc.GiftOfferError_GIFT_OFFER_ERROR_NONE)
								So(*result, ShouldResemble, gift)
							})

							Convey("And the notification sending causes an error", func() {
								notificationsClient.On("SendGiftCreationNotification", mock.Anything, mock.Anything).Return(errors.New("gift create notif error"))

								result, err := app.CreateGift(nil, gifterTUID, giftedOfferID, gifteeTUID)
								So(err, ShouldEqual, rrpc.GiftOfferError_GIFT_OFFER_ERROR_NOTIFICATION_FAILURE)
								So(*result, ShouldResemble, gift)
							})
						})

						Convey("And this is not the first gift given for the offer.", func() {
							otherClaimableGift := createClaimableGift(gifteeTUID, giftedOfferID, "bad gift id")
							giftClient.On("GetOrCreateClaimableGift", mock.Anything, &gift).Return(&otherClaimableGift, nil)

							Convey("Notifications should not be sent.", func() {
								notificationsClient.On("SendGiftCreationNotification", mock.Anything, mock.Anything).Return(errors.New("gift create notif error"))

								result, err := app.CreateGift(nil, gifterTUID, giftedOfferID, gifteeTUID)
								So(err, ShouldEqual, rrpc.GiftOfferError_GIFT_OFFER_ERROR_NONE) // Should be no error, even though the notifcation client is mocked to error.
								So(*result, ShouldResemble, gift)
							})
						})
					})

					Convey("And the client gets or creates the claimable gift, but with an error", func() {
						giftClient.On("GetOrCreateClaimableGift", mock.Anything, &gift).Return(&claimableGift, errors.New("create gift error."))

						result, err := app.CreateGift(nil, gifterTUID, giftedOfferID, gifteeTUID)
						So(err, ShouldEqual, rrpc.GiftOfferError_GIFT_OFFER_ERROR_CLAIMABLE_GIFT_CREATION_ERROR)
						So(*result, ShouldResemble, gift)
					})

					Convey("And the client fails to get or create the claimable gift.", func() {
						giftClient.On("GetOrCreateClaimableGift", mock.Anything, &gift).Return(nil, errors.New("create gift error."))

						result, err := app.CreateGift(nil, gifterTUID, giftedOfferID, gifteeTUID)
						So(err, ShouldEqual, rrpc.GiftOfferError_GIFT_OFFER_ERROR_CLAIMABLE_GIFT_CREATION_ERROR)
						So(*result, ShouldResemble, gift) // Note the gift that was created is still returned.
					})
				})

				Convey("When the client returns an error when creating the gift", func() {
					Convey("And a gift is returned despite the error", func() {
						giftClient.On("CreateGift", mock.Anything, gifterTUID, giftedOfferID, gifteeTUID).Return(&gift, errors.New("create gift error."))

						result, err := app.CreateGift(nil, gifterTUID, giftedOfferID, gifteeTUID)
						So(err, ShouldEqual, rrpc.GiftOfferError_GIFT_OFFER_ERROR_GIFT_CREATION_ERROR)
						So(*result, ShouldResemble, gift)
					})

					Convey("And a gift is not returned", func() {
						giftClient.On("CreateGift", mock.Anything, gifterTUID, giftedOfferID, gifteeTUID).Return(nil, errors.New("create gift error."))

						result, err := app.CreateGift(nil, gifterTUID, giftedOfferID, gifteeTUID)
						So(err, ShouldEqual, rrpc.GiftOfferError_GIFT_OFFER_ERROR_GIFT_CREATION_ERROR)
						So(result, ShouldBeNil)
					})
				})
			})
		})
	})
}

func TestGetGiftsFrom(t *testing.T) {
	Convey("TestGetGiftsFrom", t, func() {
		giftClient := new(mocks.IGiftingClient)
		app := GiftingImpl{
			Clients: clients.Clients{GiftingClient: giftClient},
		}

		Convey("Test GetGiftsFrom", func() {
			tuid := "123"
			tuid2 := "456"
			tuid3 := "789"
			offerID := "offer-123"

			giftID1 := "abcde"
			giftID2 := "fghk"

			gift1 := createPendingGift(giftID1, tuid, offerID, tuid2)
			gift2 := createPendingGift(giftID2, tuid, offerID, tuid3)

			gifts := make([]giftingClient.Gift, 0)
			gifts = append(gifts, gift1)
			gifts = append(gifts, gift2)

			Convey("When no offerID is provided", func() {
				Convey("When the client acts as expected", func() {
					giftClient.On("GetGiftsFrom", mock.Anything, tuid).Return(&gifts, nil)

					result, err := app.GetGiftsFrom(nil, tuid, "")
					So(err, ShouldBeNil)
					So(result, ShouldNotBeNil)
				})

				Convey("When the client returns an error", func() {
					giftClient.On("GetGiftsFrom", mock.Anything, tuid).Return(&gifts, errors.New("error of no importance."))

					_, err := app.GetGiftsFrom(nil, tuid, "")
					So(err, ShouldNotBeNil)
				})
			})

			Convey("When an offerID is provided", func() {
				Convey("When the client acts as expected", func() {
					giftClient.On("GetGiftsFromForOffer", mock.Anything, tuid, offerID).Return(&gifts, nil)

					result, err := app.GetGiftsFrom(nil, tuid, offerID)
					So(err, ShouldBeNil)
					So(result, ShouldNotBeNil)
				})

				Convey("When the client returns an error", func() {
					giftClient.On("GetGiftsFromForOffer", mock.Anything, tuid, offerID).Return(&gifts, errors.New("error of no importance."))

					_, err := app.GetGiftsFrom(nil, tuid, offerID)
					So(err, ShouldNotBeNil)
				})
			})
		})
	})
}

func TestGetGiftsTo(t *testing.T) {
	Convey("TestGetGiftsTo", t, func() {
		giftClient := new(mocks.IGiftingClient)
		app := GiftingImpl{
			Clients: clients.Clients{GiftingClient: giftClient},
		}

		Convey("Test GetGiftsTo", func() {
			tuid := "123"
			tuid2 := "456"
			tuid3 := "789"
			offerID := "offer-123"

			giftID1 := "abcde"
			giftID2 := "fghk"

			gift1 := createPendingGift(giftID1, tuid2, offerID, tuid)
			gift2 := createPendingGift(giftID2, tuid3, offerID, tuid)

			gifts := make([]giftingClient.Gift, 0)
			gifts = append(gifts, gift1)
			gifts = append(gifts, gift2)

			Convey("When no offerID is provided", func() {
				Convey("When the client acts as expected", func() {
					giftClient.On("GetGiftsTo", mock.Anything, tuid).Return(&gifts, nil)

					result, err := app.GetGiftsTo(nil, tuid, "")
					So(err, ShouldBeNil)
					So(result, ShouldNotBeNil)
				})

				Convey("When the client returns an error", func() {
					giftClient.On("GetGiftsTo", mock.Anything, tuid).Return(&gifts, errors.New("error of no importance."))

					_, err := app.GetGiftsTo(nil, tuid, "")
					So(err, ShouldNotBeNil)
				})
			})

			Convey("When an offerID is provided", func() {
				Convey("When the client acts as expected", func() {
					giftClient.On("GetGiftsToForOffer", mock.Anything, tuid, offerID).Return(&gifts, nil)

					result, err := app.GetGiftsTo(nil, tuid, offerID)
					So(err, ShouldBeNil)
					So(result, ShouldNotBeNil)
				})

				Convey("When the client returns an error", func() {
					giftClient.On("GetGiftsToForOffer", mock.Anything, tuid, offerID).Return(&gifts, errors.New("error of no importance."))

					_, err := app.GetGiftsTo(nil, tuid, offerID)
					So(err, ShouldNotBeNil)
				})
			})
		})
	})
}

func TestGetClaimableGift(t *testing.T) {
	Convey("TestGetClaimableGift", t, func() {
		giftClient := new(mocks.IGiftingClient)
		app := GiftingImpl{
			Clients: clients.Clients{GiftingClient: giftClient},
		}

		Convey("Test GetClaimableGift", func() {
			toTuid := "123"
			fromTuid := "456"
			offerID := "offer-123"
			giftID := "abcde"

			gift := createPendingGift(giftID, fromTuid, offerID, toTuid)
			claimableGift := createClaimableGift(toTuid, offerID, giftID)

			Convey("When a claimable gift is successfully returned", func() {
				Convey("And it is returned without error", func() {
					giftClient.On("GetClaimableGift", mock.Anything, toTuid, offerID).Return(&claimableGift, nil)

					Convey("And the underlying gift returns without error", func() {
						giftClient.On("GetGift", mock.Anything, giftID).Return(&gift, nil)

						claimableGift, claimableGiftErr := app.GetClaimableGift(nil, toTuid, offerID)
						So(*claimableGift, ShouldResemble, gift)
						So(claimableGiftErr, ShouldBeNil)
					})

					Convey("And the underlying gift returns with an error", func() {
						err := errors.New("Errors of minimal importance.")
						giftClient.On("GetGift", mock.Anything, giftID).Return(&gift, err)

						claimableGift, claimableGiftErr := app.GetClaimableGift(nil, toTuid, offerID)
						So(*claimableGift, ShouldResemble, gift)
						So(claimableGiftErr, ShouldEqual, err)
					})

					Convey("And the underlying gift cannot be returned", func() {
						err := errors.New("Errors of trivial importance.")
						giftClient.On("GetGift", mock.Anything, giftID).Return(nil, err)

						claimableGift, claimableGiftErr := app.GetClaimableGift(nil, toTuid, offerID)
						So(claimableGift, ShouldBeNil)
						So(claimableGiftErr, ShouldEqual, err)
					})
				})

				Convey("And it is returned with an error", func() {
					mockedClaimableGiftErr := errors.New("A claimable gift error of little importance.")
					giftClient.On("GetClaimableGift", mock.Anything, toTuid, offerID).Return(&claimableGift, mockedClaimableGiftErr)

					Convey("And the underlying gift returns without error", func() {
						giftClient.On("GetGift", mock.Anything, giftID).Return(&gift, nil)

						claimableGift, claimableGiftErr := app.GetClaimableGift(nil, toTuid, offerID)
						So(*claimableGift, ShouldResemble, gift)
						So(claimableGiftErr, ShouldEqual, mockedClaimableGiftErr)
					})

					Convey("And the underlying gift returns with an error", func() {
						err := errors.New("A gift error of downplayed importance.")
						giftClient.On("GetGift", mock.Anything, giftID).Return(&gift, err)

						claimableGift, claimableGiftErr := app.GetClaimableGift(nil, toTuid, offerID)
						So(*claimableGift, ShouldResemble, gift)
						So(claimableGiftErr, ShouldNotBeNil)
						So(claimableGiftErr.Error(), ShouldEqual, errors.Wrap(mockedClaimableGiftErr, err.Error()).Error())
					})

					Convey("And the underlying gift cannot be returned", func() {
						err := errors.New("A gift error of undermined importance.")
						giftClient.On("GetGift", mock.Anything, giftID).Return(nil, err)

						claimableGift, claimableGiftErr := app.GetClaimableGift(nil, toTuid, offerID)
						So(claimableGift, ShouldBeNil)
						So(claimableGiftErr, ShouldNotBeNil)
						So(claimableGiftErr.Error(), ShouldEqual, errors.Wrap(mockedClaimableGiftErr, err.Error()).Error())
					})
				})
			})

			Convey("When a claimable gift is not successfully returned", func() {
				err := errors.New("Errors of little importance.")

				giftClient.On("GetClaimableGift", mock.Anything, toTuid, offerID).Return(nil, err)

				claimableGift, claimableGiftErr := app.GetClaimableGift(nil, toTuid, offerID)
				So(claimableGift, ShouldBeNil)
				So(claimableGiftErr, ShouldEqual, err)
			})
		})
	})
}

func TestClaimGift(t *testing.T) {
	Convey("TestClaimGift", t, func() {
		giftClient := new(mocks.IGiftingClient)
		gatewayClient := new(gateway_mocks.Client)
		app := GiftingImpl{
			Clients: clients.Clients{GiftingClient: giftClient, GatewayClient: gatewayClient},
		}

		Convey("Test ClaimGift", func() {
			giftID := "gift-i-d"
			fromUser := "from-user"
			offerID := "off-wit-yer-heads"
			userID := "us-er-i-d"

			gift := createPendingGift(giftID, fromUser, offerID, userID)
			claimableGift := createClaimableGift(userID, offerID, giftID)

			Convey("When the gift does not exist.", func() {
				giftClient.On("GetGift", mock.Anything, giftID).Return(nil, nil)

				claimedGift, err := app.ClaimGift(nil, giftID, userID)
				So(err, ShouldEqual, rrpc.ClaimGiftedOfferError_GIFT_CLAIM_ERROR_NON_EXISTENT_GIFT)
				So(claimedGift, ShouldBeNil)
			})

			Convey("When the gift exists but if destined to a different user.", func() {
				user2 := "some-other-user"
				badGift := createPendingGift(giftID, fromUser, offerID, user2)

				giftClient.On("GetGift", mock.Anything, giftID).Return(&badGift, nil)

				claimedGift, err := app.ClaimGift(nil, giftID, userID)
				So(err, ShouldEqual, rrpc.ClaimGiftedOfferError_GIFT_CLAIM_ERROR_USER_DOES_NOT_OWN_GIFT)
				So(claimedGift, ShouldBeNil)
			})

			Convey("When the gift exists but is unclaimable.", func() {
				badGift := createPendingGift(giftID, fromUser, offerID, userID)
				badGift.Status = "JUNK"

				giftClient.On("GetGift", mock.Anything, giftID).Return(&badGift, nil)

				claimedGift, err := app.ClaimGift(nil, giftID, userID)
				So(err, ShouldEqual, rrpc.ClaimGiftedOfferError_GIFT_CLAIM_ERROR_GIFT_NOT_CLAIMABLE)
				So(claimedGift, ShouldBeNil)
			})

			Convey("When the gift exists and returns without error", func() {
				giftClient.On("GetGift", mock.Anything, giftID).Return(&gift, nil)
				Convey("When getting the claimable gift returns an error", func() {
					giftClient.On("GetClaimableGift", mock.Anything, userID, offerID).Return(&claimableGift, errors.New("An error of reduced importance."))

					claimedGift, err := app.ClaimGift(nil, giftID, userID)
					So(err, ShouldEqual, rrpc.ClaimGiftedOfferError_GIFT_CLAIM_ERROR_UNKNOWN_ERROR)
					So(claimedGift, ShouldBeNil)
				})

				Convey("When the claimable gift does not exist", func() {
					giftClient.On("GetClaimableGift", mock.Anything, userID, offerID).Return(nil, nil)

					claimedGift, err := app.ClaimGift(nil, giftID, userID)
					So(err, ShouldEqual, rrpc.ClaimGiftedOfferError_GIFT_CLAIM_ERROR_UNKNOWN_ERROR)
					So(claimedGift, ShouldBeNil)
				})

				Convey("When the claimable gift does not exist and there's an error when retrieving it", func() {
					giftClient.On("GetClaimableGift", mock.Anything, userID, offerID).Return(nil, errors.New("An error of limited importance."))

					claimedGift, err := app.ClaimGift(nil, giftID, userID)
					So(err, ShouldEqual, rrpc.ClaimGiftedOfferError_GIFT_CLAIM_ERROR_UNKNOWN_ERROR)
					So(claimedGift, ShouldBeNil)
				})

				Convey("When the claimable gift is not the same as the gift attempting to be claimed", func() {
					badGiftID := "bad-gift-i-d"
					badClaimableGift := createClaimableGift(userID, offerID, badGiftID)
					giftClient.On("GetClaimableGift", mock.Anything, userID, offerID).Return(&badClaimableGift, nil)

					claimedGift, err := app.ClaimGift(nil, giftID, userID)
					So(err, ShouldEqual, rrpc.ClaimGiftedOfferError_GIFT_CLAIM_ERROR_GIFT_NOT_CLAIMABLE)
					So(claimedGift, ShouldBeNil)
				})

				Convey("When gift attempting to be claimed does not belong to the same user as that of the claimable gift", func() {
					badUserID := "bad-user-i-d"
					badClaimableGift := createClaimableGift(badUserID, offerID, giftID)
					giftClient.On("GetClaimableGift", mock.Anything, userID, offerID).Return(&badClaimableGift, nil)

					claimedGift, err := app.ClaimGift(nil, giftID, userID)
					So(err, ShouldEqual, rrpc.ClaimGiftedOfferError_GIFT_CLAIM_ERROR_GIFT_NOT_CLAIMABLE)
					So(claimedGift, ShouldBeNil)
				})
				Convey("When the claimable gift exists and returns without an error", func() {
					giftClient.On("GetClaimableGift", mock.Anything, userID, offerID).Return(&claimableGift, nil)

					Convey("When calling gateway returns an error", func() {
						gatewayClient.On("ClaimPrimeOffer", mock.Anything, userID, offerID, mock.Anything, mock.Anything, mock.Anything).Return(nil, nil, errors.New("An error of sub-optimal importance"))

						claimedGift, err := app.ClaimGift(nil, giftID, userID)
						So(err, ShouldEqual, rrpc.ClaimGiftedOfferError_GIFT_CLAIM_ERROR_UNKNOWN_ERROR)
						So(claimedGift, ShouldBeNil)
					})

					Convey("When calling gateway returns an error response", func() {
						gatewayClient.On("ClaimPrimeOffer", mock.Anything, userID, offerID, mock.Anything, mock.Anything, mock.Anything).Return(nil, &samus_gateway.ErrorResponse{}, nil)

						claimedGift, err := app.ClaimGift(nil, giftID, userID)
						So(err, ShouldEqual, rrpc.ClaimGiftedOfferError_GIFT_CLAIM_ERROR_UNKNOWN_ERROR)
						So(claimedGift, ShouldBeNil)
					})

					Convey("When calling gateway returns a nil response", func() {
						gatewayClient.On("ClaimPrimeOffer", mock.Anything, userID, offerID, mock.Anything, mock.Anything, mock.Anything).Return(nil, nil, nil)

						claimedGift, err := app.ClaimGift(nil, giftID, userID)
						So(err, ShouldEqual, rrpc.ClaimGiftedOfferError_GIFT_CLAIM_ERROR_UNKNOWN_ERROR)
						So(claimedGift, ShouldBeNil)
					})

					Convey("When calling gateway is successful", func() {
						gatewayClient.On("ClaimPrimeOffer", mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything, mock.Anything).Return(&samus_gateway.PrimeOfferEntitlementResponse{}, nil, nil)

						Convey("When claiming the gift returns an error", func() {
							giftClient.On("ClaimGift", mock.Anything, giftID).Return(&gift, errors.New("errors of minimal importance."))

							claimedGift, err := app.ClaimGift(nil, giftID, userID)
							So(err, ShouldEqual, rrpc.ClaimGiftedOfferError_GIFT_CLAIM_ERROR_UNKNOWN_ERROR)
							So(claimedGift, ShouldNotBeNil)
							So(*claimedGift, ShouldResemble, gift) // Note that this should return the claimed gift returned by the gifting client if it returns one.
						})

						Convey("When claiming the gift is successful", func() {
							giftClient.On("ClaimGift", mock.Anything, giftID).Return(&gift, nil)

							claimedGift, err := app.ClaimGift(nil, giftID, userID)
							So(err, ShouldEqual, rrpc.ClaimGiftedOfferError_GIFT_CLAIM_ERROR_NONE)
							So(claimedGift, ShouldNotBeNil)
							So(*claimedGift, ShouldResemble, gift)
						})
					})
				})
			})
		})
	})
}
