package notifications

import (
	"context"

	"code.justin.tv/chat/pushy/client/events"
	"code.justin.tv/samus/rex/config"
	"code.justin.tv/samus/rex/internal/clients/gifting"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/sns"
	"github.com/cactus/go-statsd-client/statsd"
	"github.com/pkg/errors"
)

type INotificationsClient interface {
	SendGiftCreationNotification(ctx context.Context, gift *gifting.Gift) error
}

type pushyNotificationsClient struct {
	pushyClient events.Publisher
	offerTitle  string
}

func (c pushyNotificationsClient) SendGiftCreationNotification(ctx context.Context, gift *gifting.Gift) error {
	if gift == nil {
		return nil
	}

	return c.pushyClient.Publish(ctx, events.PrimeGiftReceivedEvent, events.PrimeGiftReceivedParams{
		FromTwitchUserID: gift.From,
		ToTwitchUserID:   gift.To,
		OfferID:          gift.OfferID,
		OfferTitle:       c.offerTitle,
	})
}

func NewNotificationsClient(config *config.Configuration, stats statsd.Statter) (INotificationsClient, error) {
	sess, err := session.NewSession(&aws.Config{Region: aws.String(config.AWSRegion)})
	snsClient := sns.New(sess)
	if err != nil {
		return nil, errors.Wrap(err, "Failed to create AWS session for notifications client.")
	}

	pushyClient, err := events.NewPublisher(events.Config{
		DispatchSNSARN: config.PushyPublishingARN,
		Stats:          stats,
		SNSClient:      snsClient,
	})
	if err != nil {
		return nil, errors.Wrap(err, "Failed to create Pushy service for notifications client.")
	}

	return &pushyNotificationsClient{
		pushyClient: pushyClient,
		offerTitle:  config.GiftingOfferTitle,
	}, nil
}
