package status

import (
	"code.justin.tv/samus/rex/config"
	rrpc "code.justin.tv/samus/rex/rpc"
	log "github.com/sirupsen/logrus"
	"golang.org/x/net/context"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/guregu/dynamo"
)

const defaultStatSampleRate = 1.0
const DEFAULT_STATUS = rrpc.OfferStatus_UNSEEN
const MAX_BATCH_SIZE = 25

type OfferStatus struct {
	TwitchUserID string `dynamo:"twitchId"`
	OfferID      string `dynamo:"offerId"`
	Status       string `dynamo:"status"`
}

var dynamoStatusToOfferStatus = map[string]rrpc.OfferStatus{
	"UNSEEN":     rrpc.OfferStatus_UNSEEN,
	"SEEN":       rrpc.OfferStatus_SEEN,
	"CLAIMED":    rrpc.OfferStatus_CLAIMED,
	"DISMISSED":  rrpc.OfferStatus_DISMISSED,
	"OVERRIDDEN": rrpc.OfferStatus_OVERRIDDEN,
}

var offerStatusToDynamoStatus = map[rrpc.OfferStatus]string{
	rrpc.OfferStatus_UNSEEN:     "UNSEEN",
	rrpc.OfferStatus_SEEN:       "SEEN",
	rrpc.OfferStatus_CLAIMED:    "CLAIMED",
	rrpc.OfferStatus_DISMISSED:  "DISMISSED",
	rrpc.OfferStatus_OVERRIDDEN: "OVERRIDDEN",
}

type IStatusClient interface {
	// GetOfferStatus returns the offer statuses for the given user and offers
	GetOfferStatuses(ctx context.Context, userID string, offerIDs []string) (map[string]rrpc.OfferStatus, error)

	UpdateOfferStatuses(ctx context.Context, userID string, offerStatuses map[string]rrpc.OfferStatus) error
}

type StatusClient struct {
	table dynamo.Table
}

func NewStatusClient(config *config.Configuration) (IStatusClient, error) {
	db := dynamo.New(session.New(), &aws.Config{Region: aws.String(config.AWSRegion)})
	table := db.Table("OfferStatus")
	return &StatusClient{
		table: table,
	}, nil
}

func (s *StatusClient) GetOfferStatuses(ctx context.Context, userID string, offerIDs []string) (map[string]rrpc.OfferStatus, error) {
	statuses := make(map[string]rrpc.OfferStatus)

	var results []OfferStatus
	var keys []dynamo.Keyed
	for _, offerID := range offerIDs {
		keys = append(keys, dynamo.Keys{userID, offerID})
	}
	err := s.table.Batch("twitchId", "offerId").
		Get(keys...).
		All(&results)
	if err != nil {
		logger := log.WithFields(log.Fields{
			"userID":   userID,
			"offerIDs": offerIDs,
		})
		logger.Error("[StatusClient.GetOfferStatuses] : ", err)
	}

	for _, offer := range results {
		if offer.Status != "" {
			if val, ok := dynamoStatusToOfferStatus[offer.Status]; ok {
				statuses[offer.OfferID] = val
				continue
			} else {
				logger := log.WithFields(log.Fields{
					"userID":       userID,
					"offerID":      offer.OfferID,
					"dynamoStatus": offer.Status,
				})
				logger.Warn("[StatusClient.GetOfferStatuses] : Unknown offer status")
			}
		}
		// default any null status
		statuses[offer.OfferID] = DEFAULT_STATUS
	}

	// Add any missing offers as default
	for _, offerID := range offerIDs {
		if _, ok := statuses[offerID]; !ok {
			statuses[offerID] = DEFAULT_STATUS
		}
	}

	return statuses, nil
}

func (s *StatusClient) UpdateOfferStatuses(ctx context.Context, userID string, offerStatuses map[string]rrpc.OfferStatus) error {
	var retErr error
	var statuses []interface{}

	for offerID, status := range offerStatuses {
		statuses = append(statuses, OfferStatus{TwitchUserID: userID, OfferID: offerID, Status: offerStatusToDynamoStatus[status]})
	}
	batches := batchStatuses(statuses, MAX_BATCH_SIZE)

	for _, batch := range batches {
		_, err := s.table.Batch("twitchId", "offerId").
			Write().
			Put(batch...).
			Run()

		if err != nil {
			logger := log.WithFields(log.Fields{
				"userID":           userID,
				"offerStatusBatch": batch,
			})
			logger.Error("[StatusClient.UpdateOfferStatuses] : ", err)
			retErr = err
		}
	}
	// return an error if any of the batches errored
	return retErr
}

func batchStatuses(statuses []interface{}, batchSize int) [][]interface{} {
	var batches [][]interface{}
	for i := 0; i < len(statuses); i += batchSize {
		end := i + batchSize

		if end > len(statuses) {
			end = len(statuses)
		}

		batches = append(batches, statuses[i:end])
	}
	return batches
}
