package metrics_test

import (
	"testing"
	"time"

	"code.justin.tv/samus/rex/metrics"
	"code.justin.tv/samus/rex/mocks"
	"github.com/aws/aws-sdk-go/service/cloudwatch"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
)

const testStage = "test"
const testMetricName = "test-metric"
const testAwsRegion = "us-west-2"
const rexMetricNamespace = "rex"
const acceptableMetricDelta = .01

type testMetric struct {
	name  string
	value time.Duration
}

func wait(ms int) {
	timer := time.After(time.Millisecond * time.Duration(ms))
	<-timer
}

func newTestMetric(name string, value time.Duration) testMetric {
	return testMetric{
		name:  name,
		value: value,
	}
}

func newTestMetrics(name string, values ...time.Duration) []testMetric {
	nValues := len(values)
	testMetrics := make([]testMetric, 0, nValues)
	for i := 0; i < nValues; i++ {
		testMetrics = append(testMetrics, newTestMetric(name, values[i]))
	}
	return testMetrics
}

func ms(ms int) time.Duration {
	msDelta := time.Millisecond * time.Duration(ms)
	return time.Since(time.Now().Add(msDelta * -1))
}

func mss(msIn ...int) []time.Duration {
	nMsIn := len(msIn)
	msOut := make([]time.Duration, 0, nMsIn)
	for i := 0; i < nMsIn; i++ {
		msOut = append(msOut, ms(msIn[i]))
	}
	return msOut
}

func logMetrics(metricLogger metrics.IMetricLogger, metrics ...testMetric) {
	for _, m := range metrics {
		metricLogger.LogDurationMetric(m.name, m.value)
	}
}

func testSetup(bufferSize int, batchSize int) (*mocks.CloudWatchAPI, metrics.IMetricLogger, metrics.IMetricFlusher) {
	return testSetupWithFlushParams(bufferSize, batchSize, time.Minute, time.Second, .75)
}

func testSetupWithFlushParams(bufferSize int, batchSize int, flushInterval time.Duration, flushPollCheckDelay time.Duration, bufferEmergencyFlushPercentage float64) (*mocks.CloudWatchAPI, metrics.IMetricLogger, metrics.IMetricFlusher) {
	config := metrics.Config{
		Stage:                          testStage,
		AwsRegion:                      testAwsRegion,
		BufferSize:                     bufferSize,
		BatchSize:                      batchSize,
		FlushInterval:                  flushInterval,
		FlushPollCheckDelay:            flushPollCheckDelay,
		BufferEmergencyFlushPercentage: bufferEmergencyFlushPercentage,
	}
	mockedCloudWatch := new(mocks.CloudWatchAPI)
	metricLogger, metricFlusher := metrics.NewFromCloudwatchClient(config, mockedCloudWatch)
	return mockedCloudWatch, metricLogger, metricFlusher
}

func mockedCloudwatchMatcher(t *testing.T, testMetrics ...testMetric) interface{} {
	expectedMetrics := make([]testMetric, 0, len(testMetrics))
	copy(expectedMetrics, testMetrics)
	return mock.MatchedBy(
		func(req *cloudwatch.PutMetricDataInput) bool {
			for i, aMetric := range expectedMetrics {
				assert.Equal(t, aMetric.name, *req.MetricData[i].MetricName, "Recieved unexpected metric name")
				assert.Equal(t, testStage, *req.MetricData[i].Dimensions[0].Value, "Recieved unexpected metric stage")
				assert.Equal(t, rexMetricNamespace, *req.Namespace, "Recieved unexpected metric namespace")
				assert.InDelta(t, aMetric.value.Seconds(), *req.MetricData[i].Value, acceptableMetricDelta, "Recieved unexpected metric value")
			}
			return true
		})
}
func TestLogAndFlushMetrics_NoMetrics(t *testing.T) {
	mockedCloudwatch, _, metricFlusher := testSetup(1, 1)
	metricFlusher.FlushMetrics()
	mockedCloudwatch.AssertNumberOfCalls(t, "PutMetricData", 0)
}

func TestLogAndFlushMetrics_SingleMetricLowBuffer(t *testing.T) {
	mockedCloudwatch, metricLogger, metricFlusher := testSetup(1, 1)
	testDuration := ms(100)
	testMetric := newTestMetric(testMetricName, testDuration)
	argumentMatcher := mockedCloudwatchMatcher(t, testMetric)

	mockedCloudwatch.On("PutMetricData", argumentMatcher).Return(nil, nil)
	logMetrics(metricLogger, testMetric)
	metricFlusher.FlushMetrics()
	mockedCloudwatch.AssertNumberOfCalls(t, "PutMetricData", 1)
}

func TestLogAndFlushMetrics_SingleMetricHighBuffer(t *testing.T) {
	mockedCloudwatch, metricLogger, metricFlusher := testSetup(1000, 1)
	testDuration := ms(100)
	testMetric := newTestMetric(testMetricName, testDuration)
	argumentMatcher := mockedCloudwatchMatcher(t, testMetric)

	mockedCloudwatch.On("PutMetricData", argumentMatcher).Return(nil, nil)
	logMetrics(metricLogger, testMetric)
	metricFlusher.FlushMetrics()
	mockedCloudwatch.AssertNumberOfCalls(t, "PutMetricData", 1)
}

func TestLogAndFlushMetrics_EvenBatching(t *testing.T) {
	// Expect 4 batches of 2 to cloudwatch
	mockedCloudwatch, metricLogger, metricFlusher := testSetup(8, 2)
	durations := mss(10, 20, 30, 40, 50, 60, 70, 80)
	testMetrics1 := newTestMetrics(testMetricName, durations[0:2]...)
	testMetrics2 := newTestMetrics(testMetricName, durations[2:4]...)
	testMetrics3 := newTestMetrics(testMetricName, durations[4:6]...)
	testMetrics4 := newTestMetrics(testMetricName, durations[6:8]...)
	allTestMetrics := newTestMetrics(testMetricName, durations...)

	argumentMatcher1 := mockedCloudwatchMatcher(t, testMetrics1...)
	argumentMatcher2 := mockedCloudwatchMatcher(t, testMetrics2...)
	argumentMatcher3 := mockedCloudwatchMatcher(t, testMetrics3...)
	argumentMatcher4 := mockedCloudwatchMatcher(t, testMetrics4...)

	mockedCloudwatch.On("PutMetricData", argumentMatcher1).Return(nil, nil)
	mockedCloudwatch.On("PutMetricData", argumentMatcher2).Return(nil, nil)
	mockedCloudwatch.On("PutMetricData", argumentMatcher3).Return(nil, nil)
	mockedCloudwatch.On("PutMetricData", argumentMatcher4).Return(nil, nil)
	logMetrics(metricLogger, allTestMetrics...)
	metricFlusher.FlushMetrics()
	mockedCloudwatch.AssertNumberOfCalls(t, "PutMetricData", 4)
}

func TestLogAndFlushMetrics_UnevenBatching(t *testing.T) {
	// Expect 2 batches of 3 and 1 batch of 1 to cloudwatch
	mockedCloudwatch, metricLogger, metricFlusher := testSetup(7, 3)
	durations := mss(10, 20, 30, 40, 50, 60, 70)
	testMetrics1 := newTestMetrics(testMetricName, durations[0:3]...)
	testMetrics2 := newTestMetrics(testMetricName, durations[3:6]...)
	testMetrics3 := newTestMetrics(testMetricName, durations[6:7]...)
	allTestMetrics := newTestMetrics(testMetricName, durations...)

	argumentMatcher1 := mockedCloudwatchMatcher(t, testMetrics1...)
	argumentMatcher2 := mockedCloudwatchMatcher(t, testMetrics2...)
	argumentMatcher3 := mockedCloudwatchMatcher(t, testMetrics3...)

	mockedCloudwatch.On("PutMetricData", argumentMatcher1).Return(nil, nil)
	mockedCloudwatch.On("PutMetricData", argumentMatcher2).Return(nil, nil)
	mockedCloudwatch.On("PutMetricData", argumentMatcher3).Return(nil, nil)
	logMetrics(metricLogger, allTestMetrics...)
	metricFlusher.FlushMetrics()
	mockedCloudwatch.AssertNumberOfCalls(t, "PutMetricData", 3)
}

func TestLogAndFlushMetrics_LogFlushRepeat(t *testing.T) {
	// Expect 4 batches of 2 to cloudwatch (twice)
	mockedCloudwatch, metricLogger, metricFlusher := testSetup(8, 2)

	// First time
	durations1 := mss(10, 20, 30, 40, 50, 60, 70, 80)
	testMetrics11 := newTestMetrics(testMetricName, durations1[0:2]...)
	testMetrics12 := newTestMetrics(testMetricName, durations1[2:4]...)
	testMetrics13 := newTestMetrics(testMetricName, durations1[4:6]...)
	testMetrics14 := newTestMetrics(testMetricName, durations1[6:8]...)
	allTestMetrics1 := newTestMetrics(testMetricName, durations1...)

	argumentMatcher11 := mockedCloudwatchMatcher(t, testMetrics11...)
	argumentMatcher12 := mockedCloudwatchMatcher(t, testMetrics12...)
	argumentMatcher13 := mockedCloudwatchMatcher(t, testMetrics13...)
	argumentMatcher14 := mockedCloudwatchMatcher(t, testMetrics14...)

	mockedCloudwatch.On("PutMetricData", argumentMatcher11).Return(nil, nil)
	mockedCloudwatch.On("PutMetricData", argumentMatcher12).Return(nil, nil)
	mockedCloudwatch.On("PutMetricData", argumentMatcher13).Return(nil, nil)
	mockedCloudwatch.On("PutMetricData", argumentMatcher14).Return(nil, nil)
	logMetrics(metricLogger, allTestMetrics1...)
	metricFlusher.FlushMetrics()
	mockedCloudwatch.AssertNumberOfCalls(t, "PutMetricData", 4)

	// Second time
	durations2 := mss(100, 200, 300, 400, 500, 600, 700, 800)
	testMetrics21 := newTestMetrics(testMetricName, durations2[0:2]...)
	testMetrics22 := newTestMetrics(testMetricName, durations2[2:4]...)
	testMetrics23 := newTestMetrics(testMetricName, durations2[4:6]...)
	testMetrics24 := newTestMetrics(testMetricName, durations2[6:8]...)
	allTestMetrics2 := newTestMetrics(testMetricName, durations2...)

	argumentMatcher21 := mockedCloudwatchMatcher(t, testMetrics21...)
	argumentMatcher22 := mockedCloudwatchMatcher(t, testMetrics22...)
	argumentMatcher23 := mockedCloudwatchMatcher(t, testMetrics23...)
	argumentMatcher24 := mockedCloudwatchMatcher(t, testMetrics24...)

	mockedCloudwatch.On("PutMetricData", argumentMatcher21).Return(nil, nil)
	mockedCloudwatch.On("PutMetricData", argumentMatcher22).Return(nil, nil)
	mockedCloudwatch.On("PutMetricData", argumentMatcher23).Return(nil, nil)
	mockedCloudwatch.On("PutMetricData", argumentMatcher24).Return(nil, nil)
	logMetrics(metricLogger, allTestMetrics2...)
	metricFlusher.FlushMetrics()
	mockedCloudwatch.AssertNumberOfCalls(t, "PutMetricData", 8)
}

func TestLogAndFlushMetrics_BufferOverflow(t *testing.T) {
	// Additional metrics logged outside the buffer space are lost
	mockedCloudwatch, metricLogger, metricFlusher := testSetup(1, 1)
	testDuration1 := ms(100)
	lostDurations := mss(4, 8, 15, 16, 23, 42)
	testMetric1 := newTestMetric(testMetricName, testDuration1)
	lostMetrics := newTestMetrics(testMetricName, lostDurations...)

	argumentMatcher1 := mockedCloudwatchMatcher(t, testMetric1)
	mockedCloudwatch.On("PutMetricData", argumentMatcher1).Return(nil, nil)
	logMetrics(metricLogger, testMetric1)
	logMetrics(metricLogger, lostMetrics...)
	metricFlusher.FlushMetrics()
	mockedCloudwatch.AssertNumberOfCalls(t, "PutMetricData", 1)

	// Now that we've flushed, additional metrics can be added
	testDuration2 := ms(200)
	testMetric2 := newTestMetric(testMetricName, testDuration2)
	argumentMatcher2 := mockedCloudwatchMatcher(t, testMetric2)
	mockedCloudwatch.On("PutMetricData", argumentMatcher2).Return(nil, nil)
	logMetrics(metricLogger, testMetric2)
	metricFlusher.FlushMetrics()
	mockedCloudwatch.AssertNumberOfCalls(t, "PutMetricData", 2)
}

func TestShouldFlushMetrics_TimeNotExpired(t *testing.T) {
	_, _, metricFlusher := testSetupWithFlushParams(4, 4, time.Minute, time.Second, .75)
	assert.False(t, metricFlusher.ShouldFlush())
}

func TestShouldFlushMetrics_TimeExpired(t *testing.T) {
	_, _, metricFlusher := testSetupWithFlushParams(4, 4, time.Millisecond*10, time.Millisecond, .75)
	wait(10)
	assert.True(t, metricFlusher.ShouldFlush())
}

func TestShouldFlushMetrics_NotAtEmergencyBufferThreshold(t *testing.T) {
	_, metricLogger, metricFlusher := testSetupWithFlushParams(4, 4, time.Minute, time.Second, .75)
	testDurations := mss(1, 2)
	testMetrics := newTestMetrics(testMetricName, testDurations...)
	logMetrics(metricLogger, testMetrics...)
	assert.False(t, metricFlusher.ShouldFlush())
}

func TestShouldFlushMetrics_AtEmergencyBufferThreshold(t *testing.T) {
	_, metricLogger, metricFlusher := testSetupWithFlushParams(4, 4, time.Minute, time.Second, .75)
	testDurations := mss(1, 2, 3)
	testMetrics := newTestMetrics(testMetricName, testDurations...)
	logMetrics(metricLogger, testMetrics...)
	assert.True(t, metricFlusher.ShouldFlush())
}
