"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var Player_1 = require("../models/Player");
var GameMap_1 = require("../models/GameMap");
var constants_1 = require("../constants");
var RoundResultsTimer_js_1 = require("../models/RoundResultsTimer.js");
var GameLogic = /** @class */ (function () {
    function GameLogic() {
    }
    /** Starts the physics simulation.
     * Setting instantResults to true will execute all the updates for the round synchronously.
     * Otherwise, this function sets up the simulation at a regular speed. */
    GameLogic.startPhysicsSimulation = function (gameState, instantResults) {
        // Get instant results on the server.
        if (instantResults) {
            for (var i = 0; i < RoundResultsTimer_js_1.getPhaseDurationTick(RoundResultsTimer_js_1.ResultsPhase.LAUNCH_PLAYERS); i++) {
                GameLogic.updateMovementOneTick(gameState);
            }
            for (var i = 0; i < RoundResultsTimer_js_1.getPhaseDurationTick(RoundResultsTimer_js_1.ResultsPhase.MAP_SHRINK); i++) {
                GameLogic.updateMapShrinkOneTick(gameState);
            }
        }
        else {
            // Start the results at regular speed on the client.
            if (!gameState.roundResultsTimer.timerIsRunning()) {
                gameState.roundResultsTimer.startTimer(gameState.getCurrentRoundResultsStartTimestamp());
            }
        }
    };
    /** Run the update functions until we are caught up to the provided timestamp.
     * This allows for clients connecting halfway-through a game to see an accurate
     * version of the game. */
    GameLogic.updateUntilGivenTime = function (gameState, time) {
        var currentTick = gameState.roundResultsTimer.getTickForTime(time);
        while (gameState.lastProcessedResultsTick < currentTick) {
            var resultsPhase = gameState.roundResultsTimer.getPhaseForTick(gameState.lastProcessedResultsTick);
            switch (resultsPhase) {
                case RoundResultsTimer_js_1.ResultsPhase.LAUNCH_PLAYERS:
                    this.updateMovementOneTick(gameState);
                    break;
                case RoundResultsTimer_js_1.ResultsPhase.MAP_SHRINK:
                    this.updateMapShrinkOneTick(gameState);
                    break;
            }
            gameState.lastProcessedResultsTick++;
        }
    };
    /** Run the movement update for one tick. */
    GameLogic.updateMovementOneTick = function (gameState) {
        // Move all players according to their velocities.
        gameState.players.forEach(function (player) {
            if (!player.dead) {
                player.currentPosition.add(player.velocity.clone().multiplyScalar(constants_1.TICK_LENGTH_SEC));
                player.velocity.multiplyScalar(1.0 - (GameLogic.FRICTION_COEFF * constants_1.TICK_LENGTH_SEC));
            }
        });
        GameLogic.handleCollisions(gameState.players);
        GameLogic.handleDeaths(gameState);
    };
    /** Run the map shrinking update for one tick. */
    GameLogic.updateMapShrinkOneTick = function (gameState) {
        gameState.gameMap.shrinkMap(GameLogic.SHRINK_PER_TICK);
        GameLogic.handleDeaths(gameState);
    };
    /** Check for any player deaths. */
    GameLogic.handleDeaths = function (gameState) {
        gameState.players.forEach(function (player) {
            var wasAlreadyDead = player.dead;
            if (!gameState.gameMap.isInSafeZone(player.currentPosition)) {
                player.dead = true;
                // Keep track of last player to die.
                if (!wasAlreadyDead) {
                    gameState.lastPlayerToDie = player;
                }
            }
        });
    };
    /** Check and handle any collisions between all players. */
    GameLogic.handleCollisions = function (players) {
        var playersAlive = players.filter(function (player) { return !player.dead; });
        for (var i = 0; i < playersAlive.length - 1; i++) {
            for (var j = i + 1; j < playersAlive.length; j++) {
                GameLogic.handlePotentialCollision(playersAlive[i], playersAlive[j]);
            }
        }
    };
    /** Returns whether the specified players are colliding. */
    GameLogic.playersAreColliding = function (player1, player2) {
        var distanceSq = player1.currentPosition.distanceSq(player2.currentPosition);
        var maxDistForCollision = Player_1.Player.PLAYER_RADIUS * 2.0;
        return distanceSq < Math.pow(maxDistForCollision, 2);
    };
    /** Check and handle a potential collision between the two specified players. */
    GameLogic.handlePotentialCollision = function (player1, player2) {
        if (!GameLogic.playersAreColliding(player1, player2)) {
            return;
        }
        var collisionPoint = player1.currentPosition.clone().add(player2.currentPosition).multiplyScalar(0.5);
        var collisionNormal = player2.currentPosition.clone().subtract(player1.currentPosition).normalize();
        // 1. Separate players to avoid collisions every tick.
        var collisionSeparationVector = collisionNormal.clone().multiplyScalar(Player_1.Player.PLAYER_RADIUS * GameLogic.COLLISION_SEPARATION_COEFF);
        player1.currentPosition.copy(collisionPoint.clone().subtract(collisionSeparationVector));
        player2.currentPosition.copy(collisionPoint.clone().add(collisionSeparationVector));
        // 2. Set new velocities using elastic collision.
        // Formula from https://en.wikipedia.org/wiki/Elastic_collision
        var v1 = player1.velocity.clone();
        var v2 = player2.velocity.clone();
        var m1 = Player_1.Player.PLAYER_MASS;
        var m2 = Player_1.Player.PLAYER_MASS;
        var x1 = player1.currentPosition.clone();
        var x2 = player2.currentPosition.clone();
        // Calculate player 1's velocity.
        var x1Subx2 = x1.clone().subtract(x2.clone());
        var v1Subv2 = v1.clone().subtract(v2);
        var dot1 = v1Subv2.dot(x1Subx2.clone()); // <v1 - v2, x1 - x2>
        var term1 = x1Subx2.clone().multiplyScalar((2 * m2 / (m1 + m2)) * (dot1 / x1Subx2.lengthSq()));
        var newVel1 = v1.clone().subtract(term1);
        player1.velocity.copy(newVel1);
        // Calculate player 2's velocity.
        var x2Subx1 = x2.clone().subtract(x1.clone());
        var v2Subv1 = v2.clone().subtract(v1);
        var dot2 = v2Subv1.dot(x2Subx1.clone()); // <v2 - v1, x2 - x1>
        var term2 = x2Subx1.clone().multiplyScalar((2 * m1 / (m2 + m1)) * (dot2 / x2Subx1.lengthSq()));
        var newVel2 = v2.clone().subtract(term2);
        player2.velocity.copy(newVel2);
    };
    GameLogic.SHRINK_PER_TICK = GameMap_1.GameMap.SHRINK_PER_ROUND / RoundResultsTimer_js_1.getPhaseDurationTick(RoundResultsTimer_js_1.ResultsPhase.MAP_SHRINK);
    // Friction coefficient to slow down player movement over time.
    GameLogic.FRICTION_COEFF = 3.3;
    // Coefficient to separate players on collision to prevent collisions from repeating every tick.
    GameLogic.COLLISION_SEPARATION_COEFF = 1.1;
    return GameLogic;
}());
exports.GameLogic = GameLogic;
