"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var constants_1 = require("../constants");
var GameMap = /** @class */ (function () {
    /** Creates a GameMap instance based on the dynamoDB object.
     * The useRoundStartValues flag indicates whether the returned GameMap instance should use
     * the round start values from the dynamoDB object for safe zone radius
     * (ex: to replay the changes on the client side). */
    function GameMap(dynamoMapState, useRoundStartValues) {
        if (useRoundStartValues === void 0) { useRoundStartValues = true; }
        this.dynamoMapState = dynamoMapState;
        this.roundStartSafeZoneRadius = useRoundStartValues ? this.dynamoMapState.roundStartSafeZoneRadius : this.dynamoMapState.roundEndSafeZoneRadius;
        this.currentSafeZoneRadius = this.roundStartSafeZoneRadius;
        this.roundEndSafeZoneRadius = this.dynamoMapState.roundEndSafeZoneRadius;
    }
    /** Create a DynamoMapState from this GameMap object to store its data in DynamoDB. */
    GameMap.prototype.createDynamoMapState = function () {
        return {
            roundStartSafeZoneRadius: this.roundStartSafeZoneRadius,
            roundEndSafeZoneRadius: this.roundEndSafeZoneRadius,
        };
    };
    /** Set the map's current safe zone radius its expected end value. Use this to reconcile the client-side
     * map radius with the expected result calculated on the server-side. */
    GameMap.prototype.reconcileWithExpectedEndState = function () {
        this.currentSafeZoneRadius = this.roundEndSafeZoneRadius;
    };
    /** Sets the expected end state to the current values. */
    GameMap.prototype.recordExpectedEndState = function () {
        this.roundEndSafeZoneRadius = this.currentSafeZoneRadius;
    };
    GameMap.prototype.shrinkMap = function (shrinkAmount) {
        this.currentSafeZoneRadius -= shrinkAmount;
    };
    GameMap.prototype.isInSafeZone = function (v) {
        return Math.abs(v.x) <= this.getSafeZoneRadius() && Math.abs(v.y) <= this.getSafeZoneRadius();
    };
    GameMap.prototype.getSafeZoneRadius = function () {
        return this.currentSafeZoneRadius;
    };
    GameMap.DEFAULT_SAFE_ZONE_RADIUS = 250;
    GameMap.SHRINK_PER_ROUND = GameMap.DEFAULT_SAFE_ZONE_RADIUS / constants_1.MAX_NUMBER_OF_ROUNDS_PER_GAME;
    return GameMap;
}());
exports.GameMap = GameMap;
