"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var Player_1 = require("./Player");
var GameMap_1 = require("./GameMap");
var Victor = require("victor");
var RoundResultsTimer_1 = require("./RoundResultsTimer");
var GameLogic_1 = require("../controllers/GameLogic");
var constants_1 = require("../constants");
var GamePhase;
(function (GamePhase) {
    GamePhase[GamePhase["STANDBY"] = 0] = "STANDBY";
    GamePhase[GamePhase["INPUT_PHASE"] = 1] = "INPUT_PHASE";
    GamePhase[GamePhase["DRUMROLL_PHASE"] = 2] = "DRUMROLL_PHASE";
    GamePhase[GamePhase["RESULTS_PHASE"] = 3] = "RESULTS_PHASE";
    GamePhase[GamePhase["GAME_FINISHED"] = 4] = "GAME_FINISHED";
})(GamePhase = exports.GamePhase || (exports.GamePhase = {}));
var GameState = /** @class */ (function () {
    /** Creates a GameState instance based on the dynamoDB object.
     * The useRoundStartValues flag indicates whether the returned GameState instance should use
     * the round start values from the dynamoDB object for safe zone radius and player states
     * (ex: to replay the changes on the client side). */
    function GameState(dynamoGameState, useRoundStartValues, roundEndCallback) {
        if (roundEndCallback === void 0) { roundEndCallback = function () { }; }
        var _this = this;
        this.dynamoGameState = dynamoGameState;
        this.lastProcessedResultsTick = -1;
        this.lastPlayerToDie = null;
        this.gamePhase = GamePhase.STANDBY;
        // Create player models.
        this.players = [];
        dynamoGameState.playerStates.forEach(function (playerState) {
            _this.players.push(new Player_1.Player(playerState, useRoundStartValues));
        });
        // Create map model.
        this.gameMap = new GameMap_1.GameMap(dynamoGameState.mapState, useRoundStartValues);
        // initial death checks
        GameLogic_1.GameLogic.handleDeaths(this);
        this.roundResultsTimer = new RoundResultsTimer_1.RoundResultsTimer(roundEndCallback);
    }
    Object.defineProperty(GameState.prototype, "gameID", {
        get: function () {
            return this.dynamoGameState.gameID;
        },
        enumerable: true,
        configurable: true
    });
    /** Get the DynamoPlayerStates from all the players in this game. */
    GameState.prototype.getDynamoPlayerStates = function () {
        var nextPlayerStates = [];
        this.players.forEach(function (player) {
            nextPlayerStates.push(player.createDynamoPlayerState());
        });
        return nextPlayerStates;
    };
    GameState.prototype.getGamePhase = function () {
        return this.gamePhase;
    };
    GameState.prototype.setGamePhase = function (newPhase) {
        this.gamePhase = newPhase;
        this.roundResultsTimer.stopTimer();
    };
    /** Increments the round number for this game state. */
    GameState.prototype.incrementRoundNumber = function () {
        this.dynamoGameState.roundNumber++;
    };
    /** Applies the provided inputs to their players. Returns true for success, false for failure. */
    GameState.prototype.applyInputToPlayer = function (opaqueUserID, input) {
        // Do not set input if player is dead or not in this game.
        // Note: it's better to have these checks here than when the user submits input because
        // the latter would require an additional DynamoDB call on input submission, which is
        // a very time-sensitive operation.
        var player = this.players.find(function (potentialPlayer) {
            return potentialPlayer.getPlayerMetadata().opaqueUserID == opaqueUserID;
        });
        if (typeof player === "undefined" || !this.gameMap.isInSafeZone(player.roundStartPosition)) {
            return false;
        }
        player.setRoundInput(input.x, input.y);
        return true;
    };
    /** Set the player's current position to its expected end value. Use this to reconcile the client-side
     * player position with the expected result calculated on the server-side. */
    GameState.prototype.reconcileWithExpectedEndState = function () {
        this.gameMap.reconcileWithExpectedEndState();
        this.players.forEach(function (player) {
            player.reconcileWithExpectedEndState();
        });
    };
    /** Sets the expected end state to the current values. */
    GameState.prototype.recordExpectedEndState = function () {
        this.gameMap.recordExpectedEndState();
        this.players.forEach(function (player) {
            player.recordExpectedEndState();
        });
    };
    /** Create a DynamoGameState from this GameState object to store its data in DynamoDB. */
    GameState.prototype.createDynamoGameState = function () {
        this.dynamoGameState.gameWinnerMetadata = this.getGameWinnerMetadata();
        this.dynamoGameState.playerStates = this.getDynamoPlayerStates();
        this.dynamoGameState.mapState = this.gameMap.createDynamoMapState();
        return this.dynamoGameState;
    };
    /** Returns the player metadata for the winning player, or null if there is no winner. */
    GameState.prototype.getGameWinnerMetadata = function () {
        // If we previously selected a winner, return that one.
        if (this.dynamoGameState.gameWinnerMetadata != null) {
            return this.dynamoGameState.gameWinnerMetadata;
        }
        // Otherwise, check to see if there is a winner.
        var playersAlive = this.players.filter(function (player) { return !player.dead; });
        if (playersAlive.length == 1) {
            return playersAlive[0].getPlayerMetadata();
        }
        else if (playersAlive.length == 0) {
            // If everybody died during the last round, return the last player to die.
            return this.lastPlayerToDie && this.lastPlayerToDie.getPlayerMetadata();
        }
        return null;
    };
    /** Places all the players at their spawn positions. */
    GameState.prototype.placePlayersAtSpawnPositions = function () {
        var numPlayers = this.players.length;
        var spawnDistanceFromCenter = GameMap_1.GameMap.DEFAULT_SAFE_ZONE_RADIUS * 0.65;
        var spawnAngleDefault = 90.0;
        var spawnAngleDelta = 360.0 / numPlayers;
        this.players.forEach(function (player, i) {
            var spawnAngle = spawnAngleDefault + spawnAngleDelta * i;
            var spawnPos = new Victor(spawnDistanceFromCenter, 0).rotateByDeg(spawnAngle);
            player.roundStartPosition.copy(spawnPos);
            player.roundEndPosition.copy(spawnPos);
        });
    };
    GameState.prototype.getRoundNumber = function () {
        return this.dynamoGameState.roundNumber;
    };
    GameState.prototype.getCurrentRoundResultsStartTimestamp = function () {
        return this.getNextRoundInputStartTimestamp() - constants_1.RESULTS_PHASE_DURATION_SEC * 1000;
    };
    GameState.prototype.getNextRoundInputStartTimestamp = function () {
        return this.getNextRoundInputCutoffTimestamp() - constants_1.INPUT_PHASE_DURATION_SEC * 1000;
    };
    GameState.prototype.getNextRoundInputCutoffTimestamp = function () {
        return this.dynamoGameState.nextInputCutoffTimestamp;
    };
    GameState.prototype.getNextRoundEarliestStatePollTimestamp = function () {
        return this.getNextRoundInputCutoffTimestamp() + constants_1.DRUMROLL_PHASE_DURATION_SEC * 1000;
    };
    return GameState;
}());
exports.GameState = GameState;
