"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var Victor = require("victor");
var Player = /** @class */ (function () {
    /** Creates a Player instance based on the dynamoDB object.
     * The useRoundStartValues flag indicates whether the returned GameState instance should use
     * the round start values from the dynamoDB object for the player position and input
     * (ex: to replay the changes on the client side). */
    function Player(dynamoPlayerState, useRoundStartValues) {
        if (useRoundStartValues === void 0) { useRoundStartValues = true; }
        this.dynamoPlayerState = dynamoPlayerState;
        // State
        this.dead = false;
        // Position
        if (useRoundStartValues) {
            this.roundStartPosition = new Victor(dynamoPlayerState.roundStartPosition.x, dynamoPlayerState.roundStartPosition.y);
            this.roundEndPosition = new Victor(dynamoPlayerState.roundEndPosition.x, dynamoPlayerState.roundEndPosition.y);
        }
        else {
            this.roundStartPosition = new Victor(dynamoPlayerState.roundEndPosition.x, dynamoPlayerState.roundEndPosition.y);
            this.roundEndPosition = new Victor(0, 0);
        }
        this.currentPosition = new Victor(this.roundStartPosition.x, this.roundStartPosition.y);
        // Movement
        var input = useRoundStartValues ? dynamoPlayerState.roundInput : { x: 0, y: 0 };
        this.roundInput = new Victor(input.x, input.y);
        this.velocity = new Victor(0, 0);
        this.calculateVelocityFromInput();
    }
    /** Create a DynamoPlayerState from this Player object to store its data in DynamoDB. */
    Player.prototype.createDynamoPlayerState = function () {
        this.dynamoPlayerState.roundStartPosition = this.roundStartPosition;
        this.dynamoPlayerState.roundEndPosition = this.roundEndPosition;
        this.dynamoPlayerState.roundInput = { x: this.getRoundInput().x, y: this.getRoundInput().y };
        return this.dynamoPlayerState;
    };
    Player.prototype.getPlayerMetadata = function () {
        return this.dynamoPlayerState.playerMetadata;
    };
    /** Set the player's current position to its expected end value. Use this to reconcile the client-side
     * player position with the expected result calculated on the server-side. */
    Player.prototype.reconcileWithExpectedEndState = function () {
        this.currentPosition.copy(this.roundEndPosition);
    };
    /** Sets the expected end state to the current values. */
    Player.prototype.recordExpectedEndState = function () {
        this.roundEndPosition.copy(this.currentPosition);
    };
    /** Calculates the player's velocity from their input. */
    Player.prototype.calculateVelocityFromInput = function () {
        this.velocity.copy(this.roundInput);
        this.velocity.multiplyScalar(Player.SPEED_SCALE);
    };
    Player.prototype.setRoundInput = function (x, y) {
        this.roundInput.x = x;
        this.roundInput.y = y;
        // Cap max player input
        if (this.roundInput.magnitude() > Player.MAX_INPUT_MAGNITUDE) {
            this.roundInput.normalize().multiplyScalar(Player.MAX_INPUT_MAGNITUDE);
        }
        this.calculateVelocityFromInput();
    };
    Player.prototype.getRoundInput = function () {
        return this.roundInput;
    };
    Player.PLAYER_RADIUS = 12;
    Player.PLAYER_MASS = 1;
    Player.SPEED_SCALE = 4.1;
    Player.MAX_INPUT_MAGNITUDE = 275;
    return Player;
}());
exports.Player = Player;
