"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var constants_1 = require("../constants");
var ResultsPhase;
(function (ResultsPhase) {
    ResultsPhase["SHOW_INPUTS"] = "SHOW_INPUTS";
    ResultsPhase["LAUNCH_PLAYERS"] = "LAUNCH_PLAYERS";
    ResultsPhase["MAP_SHRINK"] = "MAP_SHRINK";
})(ResultsPhase = exports.ResultsPhase || (exports.ResultsPhase = {}));
/** Returns the number of seconds in the specified phase. */
function getPhaseDurationSec(phase) {
    switch (phase) {
        case ResultsPhase.SHOW_INPUTS: return 1.7;
        case ResultsPhase.LAUNCH_PLAYERS: return 2.0;
        case ResultsPhase.MAP_SHRINK: return 0.5;
        default:
            console.error("Asked for duration for inexistent phase '" + phase + "'.");
            return 0;
    }
}
exports.getPhaseDurationSec = getPhaseDurationSec;
/** Returns the number of ticks in the specified phase. */
function getPhaseDurationTick(phase) {
    return Math.floor(getPhaseDurationSec(phase) / constants_1.TICK_LENGTH_SEC);
}
exports.getPhaseDurationTick = getPhaseDurationTick;
/** Calculates and returns the total duration in seconds of all results phases. */
function calculateFullResultsDurationSec() {
    var phaseKeys = Object.keys(ResultsPhase);
    var duration = 0;
    phaseKeys.forEach(function (key) {
        duration += getPhaseDurationSec(key);
    });
    return duration;
}
exports.calculateFullResultsDurationSec = calculateFullResultsDurationSec;
/** Class that keeps track of the results phase without using timeouts or intervals. */
var RoundResultsTimer = /** @class */ (function () {
    /** Create a new timer instance. */
    function RoundResultsTimer(roundEndCallback) {
        if (roundEndCallback === void 0) { roundEndCallback = function () { }; }
        this.startTime = NaN;
        this.roundEndCallback = roundEndCallback;
    }
    /** Returns the game tick corresponding to the given timestamp. */
    RoundResultsTimer.prototype.getTickForTime = function (currentTime) {
        if (currentTime === void 0) { currentTime = NaN; }
        if (Number.isNaN(currentTime)) {
            currentTime = Date.now();
        }
        if (!this.timerIsRunning()) {
            return 0;
        }
        return Math.floor((currentTime - this.startTime) / (constants_1.TICK_LENGTH_SEC * 1000.0));
    };
    /** Returns the phase which the RoundResultsTimer is currently on. */
    RoundResultsTimer.prototype.getPhaseForTime = function (currentTime) {
        if (currentTime === void 0) { currentTime = NaN; }
        return this.getPhaseForTick(this.getTickForTime(currentTime));
    };
    /** Returns the phase which the given tick would be on. */
    RoundResultsTimer.prototype.getPhaseForTick = function (tick) {
        if (!this.timerIsRunning()) {
            return null;
        }
        var ticksLeft = tick;
        var phaseKeys = Object.keys(ResultsPhase);
        for (var i = 0; i < phaseKeys.length; i++) {
            ticksLeft -= getPhaseDurationTick(phaseKeys[i]);
            if (ticksLeft < 0) {
                return phaseKeys[i];
            }
        }
        this.stopTimer();
        this.roundEndCallback();
        return phaseKeys[phaseKeys.length - 1];
    };
    RoundResultsTimer.prototype.timerIsRunning = function () {
        return !Number.isNaN(this.startTime);
    };
    /** Starts the timer, optionally providing a start timestamp to start the timer at that moment (past, present, or future). */
    RoundResultsTimer.prototype.startTimer = function (startTime) {
        if (startTime === void 0) { startTime = NaN; }
        if (!Number.isNaN(startTime)) {
            this.startTime = startTime;
        }
        else {
            this.startTime = Date.now();
        }
    };
    /** Stops the timer. */
    RoundResultsTimer.prototype.stopTimer = function () {
        this.startTime = NaN;
    };
    return RoundResultsTimer;
}());
exports.RoundResultsTimer = RoundResultsTimer;
