import { Player } from "../models/Player";
import { GameMap } from "../models/GameMap";
import { GameState } from "../models/GameState";
import { TICK_LENGTH_SEC } from "../constants";
import { ResultsPhase, getPhaseDurationTick } from "../models/RoundResultsTimer.js";


export class GameLogic {
    private static readonly SHRINK_PER_TICK = GameMap.SHRINK_PER_ROUND / getPhaseDurationTick(ResultsPhase.MAP_SHRINK);

    // Friction coefficient to slow down player movement over time.
    private static readonly FRICTION_COEFF = 3.3;

    // Coefficient to separate players on collision to prevent collisions from repeating every tick.
    private static readonly COLLISION_SEPARATION_COEFF = 1.1;

    /** Starts the physics simulation. 
     * Setting instantResults to true will execute all the updates for the round synchronously.
     * Otherwise, this function sets up the simulation at a regular speed. */
    public static startPhysicsSimulation(gameState: GameState, instantResults: boolean) {
        
        // Get instant results on the server.
        if (instantResults) {
            for (let i = 0; i < getPhaseDurationTick(ResultsPhase.LAUNCH_PLAYERS); i++) {
                GameLogic.updateMovementOneTick(gameState);
            }
            for (let i = 0; i < getPhaseDurationTick(ResultsPhase.MAP_SHRINK); i++) {
                GameLogic.updateMapShrinkOneTick(gameState);
            }
        } else {
            // Start the results at regular speed on the client.
            if (!gameState.roundResultsTimer.timerIsRunning()) {
                gameState.roundResultsTimer.startTimer(gameState.getCurrentRoundResultsStartTimestamp());
            }
        }
    }

    /** Run the update functions until we are caught up to the provided timestamp.
     * This allows for clients connecting halfway-through a game to see an accurate
     * version of the game. */
    public static updateUntilGivenTime(gameState: GameState, time: number) {
        const currentTick = gameState.roundResultsTimer.getTickForTime(time);
        while (gameState.lastProcessedResultsTick < currentTick) {
            let resultsPhase = gameState.roundResultsTimer.getPhaseForTick(gameState.lastProcessedResultsTick);
            switch (resultsPhase) {
                case ResultsPhase.LAUNCH_PLAYERS:
                    this.updateMovementOneTick(gameState);
                    break;
                case ResultsPhase.MAP_SHRINK:
                    this.updateMapShrinkOneTick(gameState);
                    break;
            }
            gameState.lastProcessedResultsTick++;
        }
    }

    /** Run the movement update for one tick. */
    private static updateMovementOneTick(gameState: GameState) {
        // Move all players according to their velocities.
        gameState.players.forEach((player) => {
            if (!player.dead) {
                player.currentPosition.add(player.velocity.clone().multiplyScalar(TICK_LENGTH_SEC))
                player.velocity.multiplyScalar(1.0 - (GameLogic.FRICTION_COEFF * TICK_LENGTH_SEC));
            }
        });

        GameLogic.handleCollisions(gameState.players);
        GameLogic.handleDeaths(gameState);
    }

    /** Run the map shrinking update for one tick. */
    private static updateMapShrinkOneTick(gameState: GameState) {
        gameState.gameMap.shrinkMap(GameLogic.SHRINK_PER_TICK);
        GameLogic.handleDeaths(gameState);
    }

    /** Check for any player deaths. */
    public static handleDeaths(gameState: GameState) {
        gameState.players.forEach((player) => {
            const wasAlreadyDead = player.dead;
            if (!gameState.gameMap.isInSafeZone(player.currentPosition)) {
                player.dead = true;
                // Keep track of last player to die.
                if (!wasAlreadyDead) {
                    gameState.lastPlayerToDie = player;
                }
            }
        })
    }

    /** Check and handle any collisions between all players. */
    private static handleCollisions(players: Player[]) {
        const playersAlive = players.filter(player => !player.dead);
        for (let i = 0; i < playersAlive.length - 1; i++) {
            for (let j = i + 1; j < playersAlive.length; j++) {
                GameLogic.handlePotentialCollision(playersAlive[i], playersAlive[j]);
            }
        }
    }

    /** Returns whether the specified players are colliding. */
    private static playersAreColliding(player1: Player, player2: Player) {
        const distanceSq = player1.currentPosition.distanceSq(player2.currentPosition);
        const maxDistForCollision = Player.PLAYER_RADIUS * 2.0;
        return distanceSq < Math.pow(maxDistForCollision, 2);
    }

    /** Check and handle a potential collision between the two specified players. */
    private static handlePotentialCollision(player1: Player, player2: Player) {
        if (!GameLogic.playersAreColliding(player1, player2)) {
            return;
        }
            
        const collisionPoint = player1.currentPosition.clone().add(player2.currentPosition).multiplyScalar(0.5);
        const collisionNormal = player2.currentPosition.clone().subtract(player1.currentPosition).normalize();

        // 1. Separate players to avoid collisions every tick.
        const collisionSeparationVector = collisionNormal.clone().multiplyScalar(Player.PLAYER_RADIUS * GameLogic.COLLISION_SEPARATION_COEFF);
        player1.currentPosition.copy( collisionPoint.clone().subtract(collisionSeparationVector));
        player2.currentPosition.copy( collisionPoint.clone().add(collisionSeparationVector));

        // 2. Set new velocities using elastic collision.
        // Formula from https://en.wikipedia.org/wiki/Elastic_collision
        const v1 = player1.velocity.clone();
        const v2 = player2.velocity.clone();
        const m1 = Player.PLAYER_MASS;
        const m2 = Player.PLAYER_MASS;
        const x1 = player1.currentPosition.clone();
        const x2 = player2.currentPosition.clone();
        
        // Calculate player 1's velocity.
        const x1Subx2 = x1.clone().subtract(x2.clone());
        const v1Subv2 = v1.clone().subtract(v2);
        const dot1 = v1Subv2.dot(x1Subx2.clone()); // <v1 - v2, x1 - x2>
        const term1 = x1Subx2.clone().multiplyScalar( (2*m2/(m1+m2)) * (dot1/x1Subx2.lengthSq()) );
        const newVel1 = v1.clone().subtract(term1);
        player1.velocity.copy(newVel1);

        // Calculate player 2's velocity.
        const x2Subx1 = x2.clone().subtract(x1.clone());
        const v2Subv1 = v2.clone().subtract(v1);
        const dot2 = v2Subv1.dot(x2Subx1.clone()); // <v2 - v1, x2 - x1>
        const term2 = x2Subx1.clone().multiplyScalar( (2*m1/(m2+m1)) * (dot2/x2Subx1.lengthSq()) );
        const newVel2 = v2.clone().subtract(term2);
        player2.velocity.copy(newVel2);
    }
}