import Victor = require('victor');
import { DynamoMapState } from "../types/DynamoTypes";
import { MAX_NUMBER_OF_ROUNDS_PER_GAME } from "../constants";

export class GameMap {
    public static readonly DEFAULT_SAFE_ZONE_RADIUS: number = 250;
    public static readonly SHRINK_PER_ROUND: number = GameMap.DEFAULT_SAFE_ZONE_RADIUS / MAX_NUMBER_OF_ROUNDS_PER_GAME;

    private roundStartSafeZoneRadius: number;
    private currentSafeZoneRadius: number;
    private roundEndSafeZoneRadius: number;
    private dynamoMapState: DynamoMapState;

    /** Creates a GameMap instance based on the dynamoDB object.
     * The useRoundStartValues flag indicates whether the returned GameMap instance should use
     * the round start values from the dynamoDB object for safe zone radius 
     * (ex: to replay the changes on the client side). */
    constructor(dynamoMapState: DynamoMapState, useRoundStartValues: boolean = true) {
        this.dynamoMapState = dynamoMapState;

        this.roundStartSafeZoneRadius = useRoundStartValues? this.dynamoMapState.roundStartSafeZoneRadius : this.dynamoMapState.roundEndSafeZoneRadius;
        this.currentSafeZoneRadius = this.roundStartSafeZoneRadius;
        this.roundEndSafeZoneRadius = this.dynamoMapState.roundEndSafeZoneRadius;
    }

    /** Create a DynamoMapState from this GameMap object to store its data in DynamoDB. */
    public createDynamoMapState(): DynamoMapState {
        return {
            roundStartSafeZoneRadius: this.roundStartSafeZoneRadius,
            roundEndSafeZoneRadius: this.roundEndSafeZoneRadius,
        }
    }

    /** Set the map's current safe zone radius its expected end value. Use this to reconcile the client-side
     * map radius with the expected result calculated on the server-side. */
    public reconcileWithExpectedEndState(): void {
        this.currentSafeZoneRadius = this.roundEndSafeZoneRadius;
    }

    /** Sets the expected end state to the current values. */
    public recordExpectedEndState(): void {
        this.roundEndSafeZoneRadius = this.currentSafeZoneRadius;
    }

    public shrinkMap(shrinkAmount: number): void {
        this.currentSafeZoneRadius -= shrinkAmount;
    }

    public isInSafeZone(v: Victor) {
        return Math.abs(v.x) <= this.getSafeZoneRadius() && Math.abs(v.y) <= this.getSafeZoneRadius();
    }

    public getSafeZoneRadius() {
        return this.currentSafeZoneRadius;
    }
}