import { Player } from "./Player";
import { GameMap } from "./GameMap";
import Victor = require('victor');
import { DynamoGameState, DynamoPlayerState, PlayerMetadata } from "../types/DynamoTypes";
import { RoundResultsTimer } from "./RoundResultsTimer";
import { GameLogic } from "../controllers/GameLogic";
import { INPUT_PHASE_DURATION_SEC, RESULTS_PHASE_DURATION_SEC, DRUMROLL_PHASE_DURATION_SEC } from "../constants";

export enum GamePhase {
    STANDBY,
    INPUT_PHASE,
    DRUMROLL_PHASE,
    RESULTS_PHASE,
    GAME_FINISHED
}

export class GameState {
    public players: Player[];
    public gameMap: GameMap;

    // Tracks the high-level phase which the game is in.
    private gamePhase: GamePhase;

    // Timer to keep track of the low-level results phase the game is in.
    public roundResultsTimer: RoundResultsTimer;
    public lastProcessedResultsTick: number;

    // Tracks the last player to die in case everyone dies in the same round.
    public lastPlayerToDie: Player | null;

    // Object to read/store in the database.
    private dynamoGameState: DynamoGameState;

    /** Creates a GameState instance based on the dynamoDB object.
     * The useRoundStartValues flag indicates whether the returned GameState instance should use
     * the round start values from the dynamoDB object for safe zone radius and player states 
     * (ex: to replay the changes on the client side). */
    constructor(dynamoGameState: DynamoGameState, useRoundStartValues: boolean, roundEndCallback: () => void = () => {}) {
        this.dynamoGameState = dynamoGameState;
        this.lastProcessedResultsTick = -1;
        this.lastPlayerToDie = null;

        this.gamePhase = GamePhase.STANDBY;

        // Create player models.
        this.players = [];
        dynamoGameState.playerStates.forEach((playerState) => {
            this.players.push(new Player(playerState, useRoundStartValues));
        })

        // Create map model.
        this.gameMap = new GameMap(dynamoGameState.mapState, useRoundStartValues);
        
        // initial death checks
        GameLogic.handleDeaths(this);


        this.roundResultsTimer = new RoundResultsTimer(roundEndCallback);
    }

    public get gameID(): string | null {
        return this.dynamoGameState.gameID;
    }

    /** Get the DynamoPlayerStates from all the players in this game. */
    private getDynamoPlayerStates(): DynamoPlayerState[] {
        let nextPlayerStates: DynamoPlayerState[] = [];
        this.players.forEach((player) => {
            nextPlayerStates.push(player.createDynamoPlayerState());
        })
        return nextPlayerStates;
    }

    public getGamePhase(): GamePhase {
        return this.gamePhase;
    }

    public setGamePhase(newPhase: GamePhase): void {
        this.gamePhase = newPhase;
        this.roundResultsTimer.stopTimer();
    }

    /** Increments the round number for this game state. */
    public incrementRoundNumber(): void {
        this.dynamoGameState.roundNumber++;
    }

    /** Applies the provided inputs to their players. Returns true for success, false for failure. */
    public applyInputToPlayer(opaqueUserID: string, input: {x: number, y: number}): boolean {
        // Do not set input if player is dead or not in this game.
        // Note: it's better to have these checks here than when the user submits input because
        // the latter would require an additional DynamoDB call on input submission, which is
        // a very time-sensitive operation.
        const player = this.players.find(function(potentialPlayer) {
            return potentialPlayer.getPlayerMetadata().opaqueUserID == opaqueUserID;
        });
        if (typeof player === "undefined" || !this.gameMap.isInSafeZone(player.roundStartPosition)) {
            return false;
        }
        player.setRoundInput(input.x, input.y);
        return true;  
    }

    /** Set the player's current position to its expected end value. Use this to reconcile the client-side
     * player position with the expected result calculated on the server-side. */
    public reconcileWithExpectedEndState(): void {
        this.gameMap.reconcileWithExpectedEndState();
        this.players.forEach((player) => {
            player.reconcileWithExpectedEndState();
        });
    }

    /** Sets the expected end state to the current values. */
    public recordExpectedEndState(): void {
        this.gameMap.recordExpectedEndState();
        this.players.forEach((player) => {
            player.recordExpectedEndState();
        });
    }

    /** Create a DynamoGameState from this GameState object to store its data in DynamoDB. */
    public createDynamoGameState(): DynamoGameState {
        this.dynamoGameState.gameWinnerMetadata = this.getGameWinnerMetadata();
        this.dynamoGameState.playerStates = this.getDynamoPlayerStates();
        this.dynamoGameState.mapState = this.gameMap.createDynamoMapState();
        return this.dynamoGameState;
    }

    /** Returns the player metadata for the winning player, or null if there is no winner. */
    private getGameWinnerMetadata(): PlayerMetadata | null {
        // If we previously selected a winner, return that one.
        if (this.dynamoGameState.gameWinnerMetadata != null) {
            return this.dynamoGameState.gameWinnerMetadata;
        }

        // Otherwise, check to see if there is a winner.
        const playersAlive = this.players.filter(player => !player.dead);
        if (playersAlive.length == 1) {
            return playersAlive[0].getPlayerMetadata();
        }
        else if (playersAlive.length == 0) {
            // If everybody died during the last round, return the last player to die.
            return this.lastPlayerToDie && this.lastPlayerToDie.getPlayerMetadata();
        }

        return null;
    }

    /** Places all the players at their spawn positions. */
    public placePlayersAtSpawnPositions(): void {
        const numPlayers = this.players.length;
        const spawnDistanceFromCenter = GameMap.DEFAULT_SAFE_ZONE_RADIUS * 0.65;
        const spawnAngleDefault = 90.0;
        const spawnAngleDelta = 360.0 / numPlayers;
        this.players.forEach((player, i) => {
            const spawnAngle = spawnAngleDefault + spawnAngleDelta * i;
            const spawnPos = new Victor(spawnDistanceFromCenter, 0).rotateByDeg(spawnAngle);
            player.roundStartPosition.copy(spawnPos);
            player.roundEndPosition.copy(spawnPos);
        })
    }

    public getRoundNumber(): number {
        return this.dynamoGameState.roundNumber;
    }

    public getCurrentRoundResultsStartTimestamp(): number {
        return this.getNextRoundInputStartTimestamp() - RESULTS_PHASE_DURATION_SEC * 1000;
    }
    public getNextRoundInputStartTimestamp(): number {
        return this.getNextRoundInputCutoffTimestamp() - INPUT_PHASE_DURATION_SEC * 1000;
    }
    public getNextRoundInputCutoffTimestamp(): number {
        return this.dynamoGameState.nextInputCutoffTimestamp;
    }
    public getNextRoundEarliestStatePollTimestamp(): number {
        return this.getNextRoundInputCutoffTimestamp() + DRUMROLL_PHASE_DURATION_SEC * 1000;
    }
}