import Victor = require('victor');
import { DynamoPlayerState, PlayerMetadata } from "../types/DynamoTypes";

export class Player {

    public static readonly PLAYER_RADIUS = 12;
    public static readonly PLAYER_MASS = 1;
    private static readonly SPEED_SCALE = 4.1;
    public static readonly MAX_INPUT_MAGNITUDE = 275;

    // Object to read/store in the database.
    private dynamoPlayerState: DynamoPlayerState;

    // State
    public dead: boolean;

    // Position
    public roundStartPosition: Victor;
    public roundEndPosition: Victor;
    public currentPosition: Victor;

    // Movement
    public velocity: Victor;
    private roundInput: Victor;

    /** Creates a Player instance based on the dynamoDB object.
     * The useRoundStartValues flag indicates whether the returned GameState instance should use
     * the round start values from the dynamoDB object for the player position and input
     * (ex: to replay the changes on the client side). */
    constructor(dynamoPlayerState: DynamoPlayerState, useRoundStartValues: boolean = true) {
        this.dynamoPlayerState = dynamoPlayerState;

        // State
        this.dead = false;

        // Position
        if (useRoundStartValues) {
            this.roundStartPosition = new Victor(
                dynamoPlayerState.roundStartPosition.x,
                dynamoPlayerState.roundStartPosition.y,
            );
            this.roundEndPosition = new Victor(
                dynamoPlayerState.roundEndPosition.x,
                dynamoPlayerState.roundEndPosition.y,
            );
        } else {
            this.roundStartPosition = new Victor(
                dynamoPlayerState.roundEndPosition.x,
                dynamoPlayerState.roundEndPosition.y,
            );
            this.roundEndPosition = new Victor(0, 0);
        }
        this.currentPosition = new Victor(this.roundStartPosition.x, this.roundStartPosition.y);
        
        // Movement
        const input = useRoundStartValues? dynamoPlayerState.roundInput : {x: 0, y: 0};
        this.roundInput = new Victor(
            input.x,
            input.y
        );
        this.velocity = new Victor(0, 0);
        this.calculateVelocityFromInput();
    }

    /** Create a DynamoPlayerState from this Player object to store its data in DynamoDB. */
    public createDynamoPlayerState(): DynamoPlayerState {
        this.dynamoPlayerState.roundStartPosition = this.roundStartPosition;
        this.dynamoPlayerState.roundEndPosition = this.roundEndPosition;
        this.dynamoPlayerState.roundInput = {x: this.getRoundInput().x, y: this.getRoundInput().y};
        return this.dynamoPlayerState;
    }

    public getPlayerMetadata(): PlayerMetadata {
        return this.dynamoPlayerState.playerMetadata;
    }

    /** Set the player's current position to its expected end value. Use this to reconcile the client-side
     * player position with the expected result calculated on the server-side. */
    public reconcileWithExpectedEndState(): void {
        this.currentPosition.copy(this.roundEndPosition);
    }

    /** Sets the expected end state to the current values. */
    public recordExpectedEndState(): void {
        this.roundEndPosition.copy(this.currentPosition);
    }

    /** Calculates the player's velocity from their input. */
    private calculateVelocityFromInput(): void {
        this.velocity.copy(this.roundInput);
        this.velocity.multiplyScalar(Player.SPEED_SCALE);
    }

    public setRoundInput(x: number, y: number): void {
        this.roundInput.x = x;
        this.roundInput.y = y;
        // Cap max player input
        if (this.roundInput.magnitude() > Player.MAX_INPUT_MAGNITUDE) {
            this.roundInput.normalize().multiplyScalar(Player.MAX_INPUT_MAGNITUDE);
        }
        this.calculateVelocityFromInput();
    }

    public getRoundInput() {
        return this.roundInput;
    }
}