import { TICK_LENGTH_SEC } from "../constants";

export enum ResultsPhase {
    SHOW_INPUTS = "SHOW_INPUTS",
    LAUNCH_PLAYERS = "LAUNCH_PLAYERS",
    MAP_SHRINK = "MAP_SHRINK",
}

/** Returns the number of seconds in the specified phase. */
export function getPhaseDurationSec(phase: ResultsPhase) {
    switch (phase) {
        case ResultsPhase.SHOW_INPUTS:    return 1.7;
        case ResultsPhase.LAUNCH_PLAYERS: return 2.0;
        case ResultsPhase.MAP_SHRINK:     return 0.5;
        default:
            console.error(`Asked for duration for inexistent phase '${phase}'.`);
            return 0;
    }
}

/** Returns the number of ticks in the specified phase. */
export function getPhaseDurationTick(phase: ResultsPhase) {
    return Math.floor(getPhaseDurationSec(phase) / TICK_LENGTH_SEC);
}

/** Calculates and returns the total duration in seconds of all results phases. */
export function calculateFullResultsDurationSec() {
    const phaseKeys = Object.keys(ResultsPhase);
    let duration = 0;
    phaseKeys.forEach((key) => {
        duration += getPhaseDurationSec(<ResultsPhase> key);
    })
    return duration;
}

/** Class that keeps track of the results phase without using timeouts or intervals. */
export class RoundResultsTimer {

    private startTime: number = NaN;
    private roundEndCallback: () => void;

    /** Create a new timer instance. */
    constructor(roundEndCallback: () => void = () => {}) {
        this.roundEndCallback = roundEndCallback;
    }

    /** Returns the game tick corresponding to the given timestamp. */
    public getTickForTime(currentTime: number = NaN): number { 
        if (Number.isNaN(currentTime)) {
            currentTime = Date.now();
        }
        if (!this.timerIsRunning()) {
            return 0;
        }
        return Math.floor((currentTime - this.startTime) / (TICK_LENGTH_SEC * 1000.0));
    }

    /** Returns the phase which the RoundResultsTimer is currently on. */
    public getPhaseForTime(currentTime: number = NaN): ResultsPhase | null {
        return this.getPhaseForTick(this.getTickForTime(currentTime));
    }

    /** Returns the phase which the given tick would be on. */
    public getPhaseForTick(tick: number): ResultsPhase | null  {
        if (!this.timerIsRunning()) {
            return null;
        }

        let ticksLeft = tick;
        const phaseKeys = Object.keys(ResultsPhase);
        for (let i = 0; i < phaseKeys.length; i++) {
            ticksLeft -= getPhaseDurationTick(<ResultsPhase> phaseKeys[i]);
            if (ticksLeft < 0) {
                return <ResultsPhase> phaseKeys[i];
            }
        }

        this.stopTimer();
        this.roundEndCallback();
        return <ResultsPhase> phaseKeys[phaseKeys.length-1];
    }

    public timerIsRunning() {
        return !Number.isNaN(this.startTime);
    }

    /** Starts the timer, optionally providing a start timestamp to start the timer at that moment (past, present, or future). */
    public startTimer(startTime: number = NaN) {
        if (!Number.isNaN(startTime)) {
            this.startTime = startTime;
        } else {
            this.startTime = Date.now();
        }
    }

    /** Stops the timer. */
    public stopTimer() {
        this.startTime = NaN;
    }
}