import { GameMap } from "../models/GameMap";
import { GameState } from "../models/GameState";
import { HALF_HIGHEST_SUPPORTED_PING, FIRST_ROUND_INPUT_PHASE_DURATION_SEC } from "../constants";

export interface DynamoGameState {
    gameID: string | null;
    gameWinnerMetadata: PlayerMetadata | null;
    roundNumber: number;
    nextInputCutoffTimestamp: number;
    mapState: DynamoMapState;
    playerStates: DynamoPlayerState[];
}

export interface DynamoMapState {
    roundStartSafeZoneRadius: number;
    roundEndSafeZoneRadius: number;
}

export interface DynamoPlayerState {
    playerMetadata: PlayerMetadata;
    roundStartPosition: {
        x: number;
        y: number;
    }
    roundEndPosition: {
        x: number;
        y: number;
    }
    roundInput: {
        x: number;
        y: number;
    }
}

export interface PlayerMetadata {
    opaqueUserID: string;
    username: string;
}

/** Creates an initial DynamoMapState object for the map in a new game. */
function createInitialDynamoMapState(): DynamoMapState {
    return ({
        roundStartSafeZoneRadius: GameMap.DEFAULT_SAFE_ZONE_RADIUS,
        roundEndSafeZoneRadius: GameMap.DEFAULT_SAFE_ZONE_RADIUS,
    });
}


/** Creates an initial DynamoPlayerState object for a player in a new game. */
function createInitialDynamoPlayerState(playerMetadata: PlayerMetadata): DynamoPlayerState {
    return ({
        playerMetadata: playerMetadata,
        roundStartPosition: {
            x: 0,
            y: 0,
        },
        roundEndPosition: {
            x: 0,
            y: 0,
        },
        roundInput: {
            x: 0,
            y: 0,
        },
    });
}

/** Creates an initial DynamoGameState for a new game. */
export function createInitialDynamoGameState(gameID: string, participatingPlayerMetadata: PlayerMetadata[]): DynamoGameState {

    const firstInputCutoffTimestamp = Date.now() + FIRST_ROUND_INPUT_PHASE_DURATION_SEC*1000 - HALF_HIGHEST_SUPPORTED_PING;
    
    const initialPlayerStates: DynamoPlayerState[] = [];
    participatingPlayerMetadata.forEach((playerMetadata) => {
        initialPlayerStates.push(createInitialDynamoPlayerState(playerMetadata));
    })

    const initialMapState = createInitialDynamoMapState();

    // Game state without the players placed at their spawn positions.
    const noPlacementDynamoGameState = {
        gameID: gameID,
        gameWinnerMetadata: null,
        roundNumber: 0,
        nextInputCutoffTimestamp: firstInputCutoffTimestamp,
        mapState: initialMapState,
        playerStates: initialPlayerStates,
    };

    const gameState: GameState = new GameState(noPlacementDynamoGameState, true);
    gameState.placePlayersAtSpawnPositions();
    return gameState.createDynamoGameState();
}