package main

import (
	"bufio"
	"fmt"
	"net/http"
	"strconv"

	"code.justin.tv/security-cop/fruit/fruitiface"
	"code.justin.tv/security-cop/fruit/fruitinmemory"
)

// NewFruitHandler returns a handler that serves fruit
func NewFruitHandler() http.Handler {
	fruits := fruitinmemory.New()

	mux := http.NewServeMux()
	mux.Handle("/get_price", &GetPriceHandler{
		fruits: fruits,
	})
	mux.Handle("/update_price", &UpdatePriceHandler{
		fruits: fruits,
	})
	return mux
}

// GetPriceHandler serves requests to get prices of fruit.
// It expects a body of the following form:
//
// <fruit>
//
// It will return the price as a body of the following form:
//
// <fruit> <price>
//
type GetPriceHandler struct {
	fruits fruitiface.FruitsAPI
}

// ServeHTTP implements http.Handler
func (h GetPriceHandler) ServeHTTP(w http.ResponseWriter, r *http.Request) {
	scanner := bufio.NewScanner(r.Body)
	scanner.Split(bufio.ScanWords)

	lines := make([]string, 0)
	for scanner.Scan() {
		if len(lines) >= 1 {
			http.Error(w, "too many words", http.StatusBadRequest)
			return
		}
		lines = append(lines, scanner.Text())
	}

	if len(lines) != 1 {
		http.Error(w, "too few words", http.StatusBadRequest)
		return
	}

	fruitName := lines[0]

	f, err := h.fruits.Get(fruitName)
	if err != nil {
		http.Error(w, err.Error(), http.StatusInternalServerError)
	}

	_, err = w.Write([]byte(fmt.Sprintf("%s %d", f.ID, f.Price)))
	if err != nil {
		http.Error(w, err.Error(), http.StatusInternalServerError)
	}
}

// UpdatePriceHandler serves requests to update the prices of fruit.
// It expects a body of the following form:
//
// <fruit> <price>
//
// and returns the same format
//
// <fruit> <price>
//
type UpdatePriceHandler struct {
	fruits fruitiface.FruitsAPI
}

// ServeHTTP implements http.Handler
func (h UpdatePriceHandler) ServeHTTP(w http.ResponseWriter, r *http.Request) {
	scanner := bufio.NewScanner(r.Body)
	scanner.Split(bufio.ScanWords)

	lines := make([]string, 0)
	for scanner.Scan() {
		if len(lines) >= 2 {
			http.Error(w, "too many words", http.StatusBadRequest)
			return
		}
		lines = append(lines, scanner.Text())
	}

	if len(lines) != 2 {
		http.Error(w, "too few words", http.StatusBadRequest)
		return
	}

	fruitName := lines[0]
	fruitPrice, err := strconv.Atoi(lines[1])
	if err != nil {
		http.Error(w, err.Error(), http.StatusInternalServerError)
	}

	err = h.fruits.UpdatePrice(fruitName, fruitPrice)
	if err != nil {
		http.Error(w, err.Error(), http.StatusInternalServerError)
	}

	_, err = w.Write([]byte(fmt.Sprintf("%s %d", fruitName, fruitPrice)))
	if err != nil {
		http.Error(w, err.Error(), http.StatusInternalServerError)
	}
}
