package main

import (
	"flag"
	"html/template"
	"log"
	"math/rand"
	"os"

	"code.justin.tv/security-cop/graphql-demo/resolver"
	"github.com/brianvoe/gofakeit"
	"github.com/graph-gophers/graphql-go"
)

const dataTemplate = `// Code generated by ./datagen. DO NOT EDIT.
package {{ .Package }}

import (
  "code.justin.tv/security-cop/graphql-demo/resolver"
	"github.com/graph-gophers/graphql-go"
)

var Comments = []*resolver.CommentFields{
{{ range .Comments -}}
{
  ID: "{{ .ID }}",
  Author: "{{ .Author }}",
  Content: "{{ .Content }}",
},
{{ end }}
}

var Posts = []*resolver.PostFields{
{{ range .Posts -}}
{
  ID: "{{ .ID }}",
  Author: "{{ .Author }}",
  Content: "{{ .Content }}",
  Comments: []graphql.ID{
    {{ range .Comments -}}
    "{{ . }}",
    {{ end }}
  },
},
{{ end }}
}

var	SponsoredPosts = []*resolver.SponsoredPostFields{
{{ range .SponsoredPosts -}}
{
  ID: "{{ .ID }}",
  Sponsor: "{{ .Sponsor }}",
  Content: "{{ .Content }}",
  Comments: []graphql.ID{
    {{ range .Comments -}}
    "{{ . }}",
    {{ end }}
  },
},
{{ end }}
}

var TimelineEntries = []*resolver.TimelineEntryFields{
{{ range .TimelineEntries -}}
{
  ID: "{{ .ID }}",
  EntryType: {{ .EntryType }},
},
{{ end }}
}

var Users = []*resolver.UserFields{
{{ range .Users -}}
{
  ID: "{{ .ID }}",
  Name: "{{ .Name }}",
},
{{ end }}
}
`

func main() {
	pkg := flag.String("package", "data", "package name")
	output := flag.String("o", "", "output")
	flag.Parse()

	users := make([]*resolver.UserFields, 100)
	for nUser := range users {
		users[nUser] = &resolver.UserFields{
			ID:   graphql.ID(gofakeit.UUID()),
			Name: gofakeit.Name(),
		}
	}

	comments := make([]*resolver.CommentFields, 0)

	posts := make([]*resolver.PostFields, 100)
	for nPost := range posts {
		posts[nPost] = &resolver.PostFields{
			ID:      graphql.ID(gofakeit.UUID()),
			Author:  users[rand.Intn(len(users))].ID,
			Content: gofakeit.HipsterSentence(10),
		}

		postCommentIDs := make([]graphql.ID, 10)
		postComments := make([]*resolver.CommentFields, 10)
		for nPostComment := range postComments {
			postCommentID := graphql.ID(gofakeit.UUID())
			postCommentIDs[nPostComment] = postCommentID
			postComments[nPostComment] = &resolver.CommentFields{
				ID:      postCommentID,
				Content: gofakeit.HipsterSentence(10),
				Author:  users[rand.Intn(len(users))].ID,
			}
		}
		posts[nPost].Comments = postCommentIDs
		comments = append(comments, postComments...)
	}

	sponsoredPosts := make([]*resolver.SponsoredPostFields, 100)
	for nSponsoredPost := range sponsoredPosts {
		sponsoredPosts[nSponsoredPost] = &resolver.SponsoredPostFields{
			ID:      graphql.ID(gofakeit.UUID()),
			Sponsor: users[rand.Intn(len(users))].ID,
			Content: gofakeit.HipsterSentence(10),
		}

		postCommentIDs := make([]graphql.ID, 10)
		postComments := make([]*resolver.CommentFields, 10)
		for nPostComment := range postComments {
			postCommentID := graphql.ID(gofakeit.UUID())
			postCommentIDs[nPostComment] = postCommentID
			postComments[nPostComment] = &resolver.CommentFields{
				ID:      postCommentID,
				Content: gofakeit.HipsterSentence(10),
				Author:  users[rand.Intn(len(users))].ID,
			}
		}
		sponsoredPosts[nSponsoredPost].Comments = postCommentIDs
		comments = append(comments, postComments...)
	}

	timelineEntries := make([]*resolver.TimelineEntryFields, len(sponsoredPosts)+len(posts))
	for nPost := range posts {
		post := posts[nPost]
		timelineEntries[nPost] = &resolver.TimelineEntryFields{
			ID:        post.ID,
			EntryType: resolver.PostTimelineEntryType,
		}
	}
	for nSponsoredPost := range sponsoredPosts {
		sponsoredPost := sponsoredPosts[nSponsoredPost]
		timelineEntries[nSponsoredPost+len(posts)] = &resolver.TimelineEntryFields{
			ID:        sponsoredPost.ID,
			EntryType: resolver.SponsoredPostTimelineEntryType,
		}
	}

	f, err := os.Create(*output)
	if err != nil {
		panic(err)
	}
	defer f.Close()

	t := template.Must(template.New("data").Parse(dataTemplate))
	if err := t.Execute(f, struct {
		Package         string
		Comments        []*resolver.CommentFields
		Posts           []*resolver.PostFields
		SponsoredPosts  []*resolver.SponsoredPostFields
		TimelineEntries []*resolver.TimelineEntryFields
		Users           []*resolver.UserFields
	}{
		Package:         *pkg,
		Comments:        comments,
		Posts:           posts,
		SponsoredPosts:  sponsoredPosts,
		TimelineEntries: timelineEntries,
		Users:           users,
	}); err != nil {
		panic(err)
	}

	log.Printf("generated %s", *output)
}
