package resolver

import (
	"flag"
	"fmt"
	"sort"
	"time"

	"github.com/graph-gophers/graphql-go"
)

var latency = flag.Duration("simulate-latency", 0, "simulated latency between requests.")

func simulateLatency() {
	time.Sleep(*latency)
}

// NewResolver returns a new resolver
func NewResolver(
	commentsData []*CommentFields,
	postsData []*PostFields,
	sponsoredPostsData []*SponsoredPostFields,
	timelineEntriesData []*TimelineEntryFields,
	usersData []*UserFields,
) *Resolver {
	comments := make(comments, len(commentsData))
	posts := make(posts, len(postsData))
	sponsoredPosts := make(sponsoredPosts, len(sponsoredPostsData))
	timelineEntries := make(timelineEntries, len(timelineEntriesData))
	users := make(users, len(usersData))

	for n := range commentsData {
		comments[n] = &comment{
			fields: *commentsData[n],
			users:  users,
		}
	}

	for n := range postsData {
		posts[n] = &post{
			fields:   *postsData[n],
			comments: comments,
			users:    users,
		}
	}

	for n := range sponsoredPostsData {
		sponsoredPosts[n] = &sponsoredPost{
			fields:   *sponsoredPostsData[n],
			comments: comments,
			users:    users,
		}
	}

	for n := range timelineEntriesData {
		timelineEntries[n] = &timelineEntry{
			fields:         *timelineEntriesData[n],
			posts:          posts,
			sponsoredPosts: sponsoredPosts,
		}
	}

	for n := range usersData {
		users[n] = &user{
			fields: *usersData[n],
		}
	}

	sort.Sort(comments)
	sort.Sort(posts)
	sort.Sort(sponsoredPosts)
	sort.Sort(timelineEntries)
	sort.Sort(users)

	return &Resolver{
		comments:        comments,
		posts:           posts,
		sponsoredPosts:  sponsoredPosts,
		timelineEntries: timelineEntries,
		users:           users,
	}
}

// Resolver implements the graphql schema
type Resolver struct {
	comments        commentsAPI
	posts           postsAPI
	sponsoredPosts  sponsoredPostsAPI
	timelineEntries timelineEntriesAPI
	users           usersAPI
}

// Posts returns all posts
func (r *Resolver) Posts(args struct {
	First int32
	After *graphql.ID
}) []*post {
	return r.posts.After(args.After).First(args.First)
}

// SponsoredPosts returns all sponsored posts
func (r *Resolver) SponsoredPosts(args struct {
	First int32
	After *graphql.ID
}) []*sponsoredPost {
	return r.sponsoredPosts.After(args.After).First(args.First)
}

// Timeline returns all timeline entries
func (r *Resolver) Timeline(args struct {
	First int32
	After *graphql.ID
}) []*timelineEntry {
	return r.timelineEntries.After(args.After).First(args.First)
}

// UpdateUser updates a user
func (r *Resolver) UpdateUser(args struct {
	ID   graphql.ID
	Name *string
}) (*user, error) {
	u, ok := r.users.Get(args.ID)
	if !ok {
		return nil, fmt.Errorf("user<%s> does not exist", args.ID)
	}
	if args.Name != nil {
		u.fields.Name = *args.Name
	}
	return u, nil
}
