require 'slack-ruby-bot'
require 'pry'
require 'open-uri'
require 'again'
require 'sqlite3'

TWITCH_CLIENT_ID = "mo0cy8y8kd7tj3ncy02olecukhg9fzg" # registered by benherr as internal for slack-bot
ENV["SLACK_API_TOKEN"] = 'YOUR TOKEN HERE' # or pass it in via the environment

class StreamOffline < Exception; end

def channel_to_bid(channel)
  open("https://api.twitch.tv/kraken/streams/%s" % channel, "Client-ID" => TWITCH_CLIENT_ID) do |s|
    data = JSON.parse(s.read)
    raise StreamOffline.new("Stream is offline") unless data["stream"]
    return data["stream"]["_id"]
  end
end

def total_viewcounts()
  open('https://api.twitch.tv/kraken/streams/summary', "Client-ID" => TWITCH_CLIENT_ID) do |body|
    data = JSON.parse(body.read)
    return data
  end
end

def info_for_bid(bid, mode)
  host = case
  when mode == "prod" then "globalviewcount-8bc9a0.sjc02.justin.tv"
  else "globalviewcount-staging.us-west2.justin.tv"
  end

  info_url = "http://%s:4565/channel_info.json?bid=%d"

  open(info_url % [host, bid]) do |url|
    data = JSON.parse(url.read)
    return data
  end
end

def info_for_channel(channel, mode)
  return info_for_bid(channel_to_bid(channel), mode)
end

def mode_report_url(channel, **options)
  url = "https://modeanalytics.com/twitch/reports/09750c5ec97b/?param_channel=%s" % channel

  #if options[:precompute] then
  #  begin
  #    open(url, "Cookie" => "_session_id=%s" % session_id) do |f|
  #      url = f.base_uri
  #    end
  #  rescue Exception => err
  #    puts err, err.backtrace
  #  end
  #end

  return url
end

def mode_reports(channels)
  channels = channels.compact.uniq
  urls = channels.map { mode_report_url(channel, precompute: channels.size <= 2) }
  return urls
end

def agg_asns(ip_info)
  asn_before_count = Hash.new(0)
  asn_after_count = Hash.new(0)
  asn_oip_tokens = Hash.new(0)
  asn_oip_ips = Hash.new(0)
  asn_rep = Hash.new { |h,k| h[k] = Array.new }
  asn_iprep_ips_filtered = Hash.new(0)
  asn_ipsnap_filtered = Hash.new(0)

  ip_info.each do |d|
    rep = d["IPRepInfo"]
    asn = rep["ASNName"]
    asn = "Unknown ASN for IP %s" % d["IP"] if asn.nil? || asn == ""
    score = rep["BlendedScore"] || rep["Score"]
    asn_iprep_ips_filtered[asn] += d["TokenCount"] if d["IPRepFrac"] == 0 && !d["IPOverused"] && !d["IPSnap"]
    asn_ipsnap_filtered[asn] += d["TokenCount"] if d["IPRepFrac"] == 0 && !d["IPOverused"] && d["IPSnap"]
    asn_rep[asn] << score
    asn_before_count[asn] += d["TokenCount"]
    asn_after_count[asn] += d["TokenCount"] * d["IPRepFrac"] * ((d["IPOverused"] && !d["IPWhitelisted"]) ? 0 : 1)
    asn_oip_tokens[asn] += d["TokenCount"] if d["IPOverused"] && !d["IPWhitelisted"]
    asn_oip_ips[asn] += 1 if d["IPOverused"] && !d["IPWhitelisted"]
  end

  asn_rep.each_key do |asn|
    rep_values = asn_rep[asn].compact
    asn_rep[asn] = rep_values.inject(&:+).to_f / rep_values.size
  end

  asn_before_count.keys.map do |asn|
    {
      name: asn,
      score: asn_rep[asn],
      before_count: asn_before_count[asn],
      after_count: asn_after_count[asn],
      overused_tokens: asn_oip_tokens[asn],
      overused_ips: asn_oip_ips[asn],
      iprep_filtered: asn_iprep_ips_filtered[asn],
      ipsnap_filtered: asn_ipsnap_filtered[asn],
    }
  end.sort_by { |x| -x[:before_count] }
end

DB = SQLite3::Database.new(File.expand_path("~/vba_score.db"))
DB.results_as_hash = true

class Viewbot < SlackRubyBot::Bot
  def self.top_channels(client, to_channel)
    total_data = total_viewcounts()
    total = total_data["viewers"].to_i
    channels = total_data["channels"].to_i

    info_url = "http://globalviewcount-staging.us-west2.justin.tv:4565/top_streams.json?limit=30"
    open(info_url) do |url|
      data = JSON.parse(url.read)

      numberify = lambda do |x|
        count = x.to_s.reverse.gsub(/(\d{3})(?=\d)/, '\\1,').reverse
        count
      end

      total_s = numberify.call(total)
      channel_s = numberify.call(channels)

      max_name_size = data.map { |channel_data| channel_data["name"].size }.max
      max_size = data.map { |channel_data| numberify.call(channel_data["channel_count_merged"]).size }.max
      run_share = 0
      idx = 0

      text = "Global CCU: %s @ %s channels\n```%s```" % [total_s, channel_s, (data.map do |channel_data|
        idx += 1
        name = channel_data["name"].rjust(max_name_size)
        name_spaces = name.delete("^ ")
        name.strip!
        url = "http://www.twitch.tv/%s" % name
        counti = channel_data["channel_count_merged"]
        new_share = counti.to_f / total * 100
        run_share += new_share
        count = numberify.call(counti).rjust(max_size)
        new_share_s = ("(%.2f" % new_share).rjust(1+2+1+2) + "%"
        run_share_s = ("%.2f" % run_share).rjust(3+1+2) + "%"

        "#%2d %s<%s|%s> has %s viewers  %s => %s)" % [idx, name_spaces, url, name, count, new_share_s, run_share_s]
      end.join("\n"))]

      client.web_client.chat_postMessage(
        channel: to_channel,
        as_user: true,

        attachments: [{
          "mrkdwn_in": ["text", "title"],
          fallback: "Top channels\n%s" % text,
          title: "Top channels",
          text: text
        }]
      )
    end
  end

  def self.process_channels(client, to_channel, channels, mode="staging")
    channels = channels.compact.uniq
    return if channels.empty?

    channels = channels.compact.uniq

    client.web_client.chat_postMessage(
      channel: to_channel,
      as_user: true,

      attachments: channels.map do |channel|
        begin
          case channel
          when /^AS(\d+)$/
            asn_id = $1
            scores = DB.execute("select * from vba_score where client_asn_id = ? order by time asc", asn_id)
            text = scores.map { |s| s["dscore"] }.join(", ")
          else
            info = info_for_channel(channel, mode)
            counts = info["ChannelCounts"]
            ip_info = info["IPInfo"]

            overused_or_token_filtered = 0

            ip_info.each do |info|
              ditched_tokens = [0, info["TokenCount"] - info["TokenLimit"]].max
              if not info["IPOverused"] then
                overused_or_token_filtered += ditched_tokens if info["IPRepFrac"] == 1
              else
                overused_or_token_filtered += info["TokenCount"] if info["IPRepFrac"] == 1
              end
            end

            asns = agg_asns(ip_info)
            total_count = 0
            total_ipsnap = 0

            asn_text = asns.map do |asn|
              total_count += asn[:before_count]
              overused = "-%d overused by %d IPs" % [asn[:overused_tokens], asn[:overused_ips]] if asn[:overused_tokens] > 0
              iprep_filtered = "-%d by IPRep" % [asn[:iprep_filtered]] if asn[:iprep_filtered] > 0 
              ipsnap_filtered = "-%d by IPSnap" % [asn[:ipsnap_filtered]] if asn[:ipsnap_filtered] > 0
              total_ipsnap += asn[:ipsnap_filtered]
              extra_info = [iprep_filtered, overused, ipsnap_filtered].compact.join("; ")
              extra_info = "[%s]" % extra_info if extra_info.size > 0
               "%4d /%4d   %s  %s" % [asn[:after_count], asn[:before_count], asn[:name], extra_info]
            end.first(50).join("\n")

            viewer_info = "Viewers: %d merged, %d after, %d before filtering, %d raw, %.3f raw/filtered; %d filtered by overused IP or tokens/IP; %d filtered by IPSnap" % [
              counts["channel_count_merged"],
              counts["channel_count_after_ip_rep"],
              counts["channel_count"],
              counts["channel_count_raw"],
              counts["channel_count_raw"].to_f / counts["channel_count_after_ip_rep"].to_f,
              overused_or_token_filtered,
              total_ipsnap
            ]

            text = [viewer_info, "", "```%s```" % asn_text].join("\n")
          end
        rescue Exception => error
          case error
          when StreamOffline then
            mode_url = mode_report_url(channel, precompute: channels.size <= 2)
            text = "Error: %s; try %s" % [error.to_s, mode_url]
          else
            puts error, error.backtrace
            text = "Error: %s" % error.to_s
          end
        end

        {
          "mrkdwn_in": ["text", "title"],
          fallback: "ASN info for %s\n%s" % [channel, viewer_info],
          title: "ASN info for %s" % channel,
          text: text
        }
      end
    )
  end

  match %r{https://twitter.com/BotDetectorBot/\S+} do |client, data, match|
    tweet_id = match.to_s[%r{status/(\d+)}, 1]
    #tweet = twitter.status(tweet_id)
    #channels = tweet.urls.map { |u| u.expanded_url[%r{twitch.tv/(\S+)}, 1] }
    process_channels(client, data.channel, channels)
  end

  scan %r{twitch.tv/([^/|>]+)/?} do |client, data, channels|
    if data.user != client.self.id and noisy_channel?(client, data.channel) then
      process_channels(client, data.channel, channels)
    end
  end

  def self.noisy_channel?(client, channel)
    return true if channel.start_with?("D")
    channel_info = client.web_client.channels_info(channel: channel)
    return channel_info.name == "viewbots-internal"
  end

  match %r{^(?<bot>[[:alnum:][:punct:]@<>]*)\s+(?<expression>.+?)$} do |client, data, match|
    # Don't react to ourselves
    direct_msg = data.channel.start_with?("D")
    should_reply = data.user != client.self.id
    should_reply &&= direct_msg || data.text.start_with?("<@")

    if should_reply then
      case match['expression'] || ''
      when /^!top$/ then
        top_channels(client, data.channel)
      when /^mode\s+(.+?)$/ then
        msg = $1.split(" ").map { |channel| mode_report_url(channel) }.join("\n")
        client.say(text: msg, channel: data.channel)
      else
        channels = match['expression'].split
        mode = "staging"
        
        if channels.first == "prod" then
          channels.shift
          mode = "prod"
        end

        process_channels(client, data.channel, channels, mode)
      end
    end
  end
end

class MyApp < SlackRubyBot::App
  on 'hello' do |client, data|
    client.logger.level = Logger::WARN
  end

  on 'message' do |client, data|
    if data.user != client.self.id then
      data.fetch('attachments', []).each do |att|
        text = att.text || att.fallback || ""
        if att.author_subname="@BotDetectorBot" || Viewbot.noisy_channel?(data.channel) then
          Viewbot.process_channels(client, data.channel, text.scan(%r{twitch.tv/([^/|>]+)/?}).flatten)
        end
      end
    end
  end
end

if Again.reloaded? then
  puts "Reloaded"
end

if __FILE__ == $0 then
  MyApp.instance.run
end
