# for every region, we create a vpc. top level supplies
# a range and we carve it up past that. currently assuming max 6 AZs (us-east-1)

locals {
  # DNS funtimes.
  # if we have dns_domain, use that. else use the combination
  # of dns_comain_infix.dns_domain_suffix
  dns_suffix = var.dns_domain_suffix != "" ? var.dns_domain_suffix : "compute.internal"

  dns_infix  = var.dns_domain_infix != "" ? var.dns_domain_infix : data.aws_region.current.name
  dns_domain = var.dns_domain != "" ? var.dns_domain : "${local.dns_infix}.${local.dns_suffix}"

  # split the network ranges in four pieces so I don't have to think _real hard_ about network aggregation.
  # this can be revisited with 0.12 and dynamic blocks.

  # current theory is - divide subnet into 4 chunks. the last chunk is reserved for public subnets.
  # the first three chunks are for private subnet usage. the list is accessed with modulo math so
  # that if we have more than 3 subnets, we flip between each "half" of a subnet there.
  # this allows for 3-6 subnets at the cost of unused ip space for the 4, 5 subnet models.
  regions           = length(local.azset)
  public_cidr_block = cidrsubnet(var.vpc_cidr, 2, 3)
  private_cidr_block = [
    cidrsubnet(var.vpc_cidr, 2, 0),
    cidrsubnet(var.vpc_cidr, 2, 1),
    cidrsubnet(var.vpc_cidr, 2, 2),
  ]

  # configure that modulo math here.
  private_cidr_size = local.regions > 3 ? 1 : 0
  private_cidr_mod  = local.regions > 3 ? 2 : 1
}

data "aws_availability_zones" "all" {
}

data "aws_region" "current" {
}

resource "aws_vpc" "vpc" {
  cidr_block = var.vpc_cidr
  tags       = module.taglib.tags

  lifecycle {
    prevent_destroy = true
  }

  enable_dns_hostnames = true
}

resource "aws_internet_gateway" "igw" {
  vpc_id = aws_vpc.vpc.id
  tags   = module.taglib.tags
}

# if we need a vpn gateway, create it here
resource "aws_vpn_gateway" "vgw" {
  count           = var.enable_vgw == "true" ? 1 : 0
  vpc_id          = aws_vpc.vpc.id
  tags            = module.taglib.tags
  amazon_side_asn = 64555
}

# DHCP - we're actually pretty close to the default...
resource "aws_vpc_dhcp_options" "default_dhcp_options" {
  domain_name         = local.dns_domain
  domain_name_servers = ["AmazonProvidedDNS"] # vpc resolver
  ntp_servers         = ["169.254.169.123"]   # vpc time server
  tags                = module.taglib.tags
}

resource "aws_vpc_dhcp_options_association" "default_dhcp_options_association" {
  vpc_id          = aws_vpc.vpc.id
  dhcp_options_id = aws_vpc_dhcp_options.default_dhcp_options.id
}

# create a subnet pair for every AZ
resource "aws_subnet" "private" {
  cidr_block = cidrsubnet(
    element(local.private_cidr_block, count.index),
    local.private_cidr_size,
    count.index % local.private_cidr_mod,
  )
  vpc_id = aws_vpc.vpc.id
  count  = local.az_count
  availability_zone = join(
    "",
    [
      data.aws_region.current.name,
      lower(element(local.azset, count.index)),
    ],
  )
  tags = merge(
    module.taglib.tags,
    {
      "Name" = "Private - ${element(local.azset, count.index)}"
    },
  )

  lifecycle {
    prevent_destroy = false
  }
}

# you need public because that's where the NAT box lives
resource "aws_subnet" "public" {
  # if you change the length here update the public and private cidr block locals!
  cidr_block              = cidrsubnet(local.public_cidr_block, 3, count.index)
  map_public_ip_on_launch = var.default_internet_public_map
  vpc_id                  = aws_vpc.vpc.id
  count                   = local.az_count
  availability_zone = join(
    "",
    [
      data.aws_region.current.name,
      lower(element(local.azset, count.index)),
    ],
  )
  tags = merge(
    module.taglib.tags,
    {
      "Name" = "Public - ${element(local.azset, count.index)}"
    },
  )

  lifecycle {
    prevent_destroy = false
  }
}

# and route tables for all the private subnets (each gets a separate NAT)
resource "aws_route_table" "private_rt" {
  vpc_id           = aws_vpc.vpc.id
  count            = local.az_count
  tags             = module.taglib.tags
  propagating_vgws = compact([join("", aws_vpn_gateway.vgw.*.id)])
}

# adopt the VPC route table for the public subnets
resource "aws_default_route_table" "vpc" {
  default_route_table_id = aws_vpc.vpc.default_route_table_id
  tags                   = module.taglib.tags
  propagating_vgws       = compact([join("", aws_vpn_gateway.vgw.*.id)])
}

# use resources, not inline, to add the default route, and s3 endpoint.
resource "aws_vpc_endpoint" "s3" {
  route_table_ids = concat(
    [aws_default_route_table.vpc.id],
    aws_route_table.private_rt.*.id,
  )
  vpc_id       = aws_vpc.vpc.id
  service_name = "com.amazonaws.${data.aws_region.current.name}.s3"
  tags         = module.taglib.tags
}

resource "aws_route" "inet4" {
  route_table_id         = aws_default_route_table.vpc.id
  destination_cidr_block = "0.0.0.0/0"
  gateway_id             = aws_internet_gateway.igw.id
}

# create a NAT gw for each AZ...
resource "aws_eip" "nat" {
  vpc   = true
  count = local.az_count
  tags  = module.taglib.tags
}

resource "aws_nat_gateway" "nat" {
  count         = local.az_count
  allocation_id = element(aws_eip.nat.*.id, count.index)
  subnet_id     = element(aws_subnet.public.*.id, count.index)
  tags          = module.taglib.tags
}

# and plug that in to the private subnet RT.
resource "aws_route" "nat" {
  count                  = local.az_count
  route_table_id         = element(aws_route_table.private_rt.*.id, count.index)
  nat_gateway_id         = element(aws_nat_gateway.nat.*.id, count.index)
  destination_cidr_block = "0.0.0.0/0"
}

# which finally! get associated back to subnets
resource "aws_route_table_association" "private_rt" {
  count          = local.az_count
  subnet_id      = element(aws_subnet.private.*.id, count.index)
  route_table_id = element(aws_route_table.private_rt.*.id, count.index)
}

# this security group is only for the vpc endpoints, though. do not add random
# instances here.
resource "aws_security_group" "ec2api" {
  name        = "ec2api"
  description = "vpc endpoints (aws apis) NOT MACHINES"
  vpc_id      = aws_vpc.vpc.id

  ingress {
    from_port   = 0
    to_port     = 0
    protocol    = -1
    cidr_blocks = [aws_vpc.vpc.cidr_block]
  }

  tags = module.taglib.tags
}

# create SSM needed endpoints
resource "aws_vpc_endpoint" "ssm" {
  vpc_endpoint_type   = "Interface"
  private_dns_enabled = true
  vpc_id              = aws_vpc.vpc.id
  service_name        = "com.amazonaws.${data.aws_region.current.name}.ssm"
  security_group_ids  = [aws_security_group.ec2api.id]
  tags                = module.taglib.tags
  subnet_ids          = aws_subnet.private.*.id
}

resource "aws_vpc_endpoint" "ssmmessages" {
  vpc_endpoint_type   = "Interface"
  private_dns_enabled = true
  vpc_id              = aws_vpc.vpc.id
  service_name        = "com.amazonaws.${data.aws_region.current.name}.ssmmessages"
  security_group_ids  = [aws_security_group.ec2api.id]
  tags                = module.taglib.tags
  subnet_ids          = aws_subnet.private.*.id
}

resource "aws_vpc_endpoint" "ec2" {
  vpc_endpoint_type   = "Interface"
  private_dns_enabled = true
  vpc_id              = aws_vpc.vpc.id
  service_name        = "com.amazonaws.${data.aws_region.current.name}.ec2"
  security_group_ids  = [aws_security_group.ec2api.id]
  tags                = module.taglib.tags
  subnet_ids          = aws_subnet.private.*.id
}

resource "aws_vpc_endpoint" "ec2messages" {
  vpc_endpoint_type   = "Interface"
  private_dns_enabled = true
  vpc_id              = aws_vpc.vpc.id
  service_name        = "com.amazonaws.${data.aws_region.current.name}.ec2messages"
  security_group_ids  = [aws_security_group.ec2api.id]
  tags                = module.taglib.tags
  subnet_ids          = aws_subnet.private.*.id
}
