provider "aws" {
  region  = var.region
  profile = var.account
}

# Optional separate DNS account
provider "aws" {
  alias   = "instance-dns-provider"
  region  = var.region
  profile = var.r53_profile
}

resource "aws_lb" "lb" {
  name               = "${var.service}-${var.environment}"
  load_balancer_type = "application"
  internal           = true
  security_groups    = var.security_groups
  subnets            = var.subnets

  tags = {
    Name        = "${var.service}-${var.environment}-alb"
    Environment = var.environment
    Owner       = var.owner
    Service     = var.service
  }
}

resource "aws_lb_target_group" "lb_target_group" {
  name     = "${var.service}-${var.environment}"
  port     = var.instance_port
  protocol = var.instance_protocol
  vpc_id   = var.vpc_id

  health_check {
    interval            = var.healthcheck_interval
    path                = var.healthcheck_path
    healthy_threshold   = var.healthcheck_healthy_threshold
    unhealthy_threshold = var.healthcheck_unhealthy_threshold
    matcher             = var.healthcheck_matcher
  }

  tags = {
    Name        = "${var.service}-${var.environment}-tg"
    Environment = var.environment
    Owner       = var.owner
    Service     = var.service
  }
}

resource "aws_lb_listener" "lb_https_listener" {
  load_balancer_arn = aws_lb.lb.arn
  port              = "443"
  protocol          = "HTTPS"
  ssl_policy        = "ELBSecurityPolicy-2016-08"
  certificate_arn   = var.ssl_certificate_arn

  default_action {
    type             = "forward"
    target_group_arn = aws_lb_target_group.lb_target_group.arn
  }
}

# Rewrite http -> https
resource "aws_lb_listener" "lb_http_listener" {
  load_balancer_arn = aws_lb.lb.arn
  port              = "80"
  protocol          = "HTTP"

  default_action {
    type = "redirect"

    redirect {
      port        = "443"
      protocol    = "HTTPS"
      status_code = "HTTP_301"
    }
  }
}

resource "aws_lb_target_group_attachment" "lb_target_group_attachment" {
  count = var.instance_count

  target_group_arn = aws_lb_target_group.lb_target_group.arn
  target_id        = element(var.instance_ids, count.index)
  port             = var.instance_port
}

# DNS
resource "aws_route53_record" "lb_dns" {
  provider = aws.instance-dns-provider
  zone_id  = var.r53_zone_id
  name     = var.service
  type     = "A"

  alias {
    name                   = aws_lb.lb.dns_name
    zone_id                = aws_lb.lb.zone_id
    evaluate_target_health = false
  }
}
