# AWS Auto Scaling Group (ASG) Terraform Module

Simple AWS Auto Scaling Group module.

* This is simple, only specifies min/max instances. Does not support dynamic scaling (yet).
* Automatically manage instance hostname DNS.
* Use latest devinfra xenial consulfree base AMI.
* Bootstrap instances to be run with puppet.

# Requirements

Requires `terraform` version `0.12.3` and greater.

# Usage

Example:

```hcl
module "asg" {
  source = "git@git-aws.internal.justin.tv:terraform-modules/aws-autoscaling.git?ref=v1.0.0"

  service            = "video-puppet-master"
  region             = "us-west-2"
  env                = "prod"
  route53_zone_id    = "Z2BV3B6X4MHC7A"
  vpc_id             = "vpc-38ab5f41"
  user_data          = templatefile("${path.module}/templates/user_data.sh.tmpl", {
                         zone_name = "us-west-2.prod.puppet.live-video.a2z.com"
                       })
  owner              = "video-coreservices@twitch.tv"
  puppet_team        = "vidcs"
  min_size           = 5
  max_size           = 5
  security_group_ids = ["sg-9123b7ef", "sg-992sgeef"]
  subnet_ids         = ["subnet-12587177", "subnet-461de11c", "subnet-50f93f7c"]
  instance_type      = "c5.mlarge"
  puppet_server      = "video-puppet.prod.puppet.live-video.a2z.com"
  puppet_ca_server   = "video-puppet-ca.prod.puppet.live-video.a2z.com"

  target_group_arns = [
    "arn:aws:elasticloadbalancing:us-west-2:152548578290:targetgroup/video-puppet-master-http/f2e0c36443240fa8",
    "arn:aws:elasticloadbalancing:us-west-2:152548578290:targetgroup/video-puppet-master-tcp/f8b08ac487a43e85"
  ]

  ebs_block_device = [{
    device_name = "/dev/xvdf"
    volume_type = "gp2"
    volume_size = 100
  }]

  tags = {
    Source = "video-coreservices/tf-io-puppet"
  }
}
```

<!-- BEGINNING OF PRE-COMMIT-TERRAFORM DOCS HOOK -->
## Inputs

| Name | Description | Type | Default | Required |
|------|-------------|:----:|:-----:|:-----:|
| asg\_depends\_on | A list of terraform explicit dependencies to be passed to the ASG | list(string) | `[]` | no |
| ebs\_block\_device | Additional EBS block devices to attach to the instance | list(map(string)) | `[]` | no |
| env | The environment of this service. Either dev, stage, prod | string | `"dev"` | no |
| health\_check\_type | 'EC2' or 'ELB'. Controls how health checking is done | string | `"EC2"` | no |
| instance\_type | The size of EC2 instance to launch | string | `"t2.micro"` | no |
| max\_size | The maximum size of the autoscale group | number | `"1"` | no |
| min\_size | The minimum size of the autoscale group | number | `"1"` | no |
| owner | The team email address of the owner of the service | string | n/a | yes |
| puppet\_ca\_server | The Puppet CA server to run puppet client against | string | `"video-puppet-ca.prod.puppet.live-video.a2z.com"` | no |
| puppet\_server | The puppet master to run puppet client against | string | `"video-puppet.prod.puppet.live-video.a2z.com"` | no |
| puppet\_team | aka twitch_team. Defaults to owner if not specified | string | `""` | no |
| region | The AWS region this ASG will be running in | string | n/a | yes |
| route53\_zone\_id | The route53 zone id that will create DNS names | string | n/a | yes |
| security\_group\_ids | A list of associated security group ids | list(string) | n/a | yes |
| service | The name of the service | string | n/a | yes |
| subnet\_ids | A list of subnet IDs to launch resources in | list(string) | n/a | yes |
| tags | Common tags to be passed to resources | map(string) | `{}` | no |
| target\_group\_arns | A list of aws_alb_target_group ARNs, for use with Application or Network Load Balancing | list(string) | `[]` | no |
| user\_data | The user data to provide when launching the instance. Do not pass gzip-compressed data via this argument | string | `""` | no |
| user\_data\_post\_puppet | The user data to be run post puppet agent run | string | `""` | no |
| vpc\_id | The VPC id this ASG will be running in | string | n/a | yes |

## Outputs

| Name | Description |
|------|-------------|
| asg | The created autoscaling group attributes |
| iam\_role\_name | The IAM role that EC2 instances will be assuming |

<!-- END OF PRE-COMMIT-TERRAFORM DOCS HOOK -->

# Development

* Setup [`pre-commit`](https://pre-commit.com/#install).
* Make sure the following dependencies are installed:
  * [`pre-commit terraform`](https://github.com/antonbabenko/pre-commit-terraform#step-1)
  * [`terraform-docs`](https://github.com/segmentio/terraform-docs#installation)
