data "aws_route53_zone" "zone" {
  zone_id = var.route53_zone_id
}

locals {
  # Trim trailing .
  zone_name = replace(data.aws_route53_zone.zone.name, "/\\.$/", "")
}

data "template_cloudinit_config" "userdata" {
  gzip          = false
  base64_encode = false

  part {
    filename     = "init.cfg"
    content_type = "text/cloud-config"

    content = templatefile("${path.module}/templates/cloudinit.yaml.tmpl", {
      service    = var.service
      dns_region = var.region
      dns_zone   = replace(local.zone_name, "${var.region}.", "")
    })
  }

  # Pre setup
  part {
    content_type = "text/x-shellscript"

    content = templatefile("${path.module}/templates/00-startup.sh.tmpl", {
      environment      = local.environment
      owner            = local.puppet_team
      service          = var.service
      dns_region       = var.region
      puppet_server    = var.puppet_server
      puppet_ca_server = var.puppet_ca_server
    })
  }

  # Custom setup pre-puppet
  part {
    content_type = "text/x-shellscript"

    content = var.user_data
  }

  # Post setup
  part {
    content_type = "text/x-shellscript"

    content = templatefile("${path.module}/templates/99-startup.sh.tmpl", {})
  }

  # Custom setup post-puppet
  part {
    content_type = "text/x-shellscript"
    content      = var.user_data_post_puppet
  }
}

resource "aws_iam_instance_profile" "instance_profile" {
  name = "${var.service}-${var.env}"
  role = aws_iam_role.role.name
}

data "aws_iam_policy_document" "instance_assume_role_policy" {
  statement {
    actions = ["sts:AssumeRole"]

    principals {
      type        = "Service"
      identifiers = ["ec2.amazonaws.com"]
    }
  }
}

resource "aws_iam_role" "role" {
  name               = "${var.service}-${var.env}"
  assume_role_policy = data.aws_iam_policy_document.instance_assume_role_policy.json

  tags = merge(local.common_tags, var.tags)
}

data "aws_ami" "base_ami" {
  most_recent = true
  owners      = ["043714768218"]

  filter {
    name   = "name"
    values = ["devtools-baseami-xenial-consulfree-*"]
  }
}

resource "aws_launch_configuration" "launch_conf" {
  name_prefix          = "${var.service}_launch_config"
  image_id             = data.aws_ami.base_ami.id
  instance_type        = var.instance_type
  iam_instance_profile = aws_iam_instance_profile.instance_profile.name
  security_groups      = var.security_group_ids

  user_data = data.template_cloudinit_config.userdata.rendered

  dynamic "ebs_block_device" {
    for_each = var.ebs_block_device
    content {
      delete_on_termination = lookup(ebs_block_device.value, "delete_on_termination", null)
      device_name           = ebs_block_device.value.device_name
      encrypted             = lookup(ebs_block_device.value, "encrypted", null)
      iops                  = lookup(ebs_block_device.value, "iops", null)
      no_device             = lookup(ebs_block_device.value, "no_device", null)
      snapshot_id           = lookup(ebs_block_device.value, "snapshot_id", null)
      volume_size           = lookup(ebs_block_device.value, "volume_size", null)
      volume_type           = lookup(ebs_block_device.value, "volume_type", null)
    }
  }

  lifecycle {
    create_before_destroy = true
  }
}

resource "aws_autoscaling_group" "asg" {
  name                 = "${var.service}_asg"
  vpc_zone_identifier  = var.subnet_ids
  min_size             = var.min_size
  max_size             = var.max_size
  health_check_type    = var.health_check_type
  launch_configuration = aws_launch_configuration.launch_conf.name
  target_group_arns    = var.target_group_arns

  depends_on = [var.asg_depends_on]

  dynamic "tag" {
    for_each = merge({
      Name                   = var.service
      "asg:hostname_pattern" = "${var.service}-#instanceid.${local.zone_name}@${var.route53_zone_id}"
    }, local.common_tags, var.tags)

    content {
      key                 = tag.key
      value               = tag.value
      propagate_at_launch = true
    }
  }
}
