# Use twitch-aws for DNS purposes
provider "aws" {
  alias = "twitch-aws"
  region = "us-west-2"
  profile = "twitch-aws"
}

provider "aws" {
  alias = "${var.account}"
  region = "${var.region}"
  profile = "${var.account}"
}

variable "template_name" {
  default = "user_data.tpl"
}

# Load user_data
data "template_file" "user_data" {
  template = "${file(join("/", list(path.module, var.template_name)))}"
  vars {
    role = "${var.role}"
    owner = "${var.owner}"
    env = "${lookup(var.environment_shortname, var.environment)}"
    environment = "${var.environment}"
    puppet = "${lookup(var.puppet_server, var.puppet)}"
    dns_region = "${lookup(var.dns_region, var.region)}"
  }
}

# AWS Instances
resource "aws_instance" "instance" {
  provider = "aws.${var.account}"
  ami = "${var.ami}"
  count = "${var.count}"
  instance_type = "${var.instance_type}"
  vpc_security_group_ids = ["${var.vpc_security_group_ids}"]
  # Stripe instances across the specified subnets
  subnet_id = "${element(var.subnet_ids, count.index % length(var.subnet_ids))}"
  root_block_device {
    volume_size = "${var.root_block_device_size}"
  }
  iam_instance_profile = "${var.instance_profile}"
  # Render and pass in the user_data template
  user_data = "${data.template_file.user_data.rendered}"
  tags {
    Name = "${var.role}-${var.environment}"
    Environment = "${lookup(var.environment_tagname, var.environment)}"
    Owner = "${var.owner}@twitch.tv"
    Service = "${var.owner}/${var.service}/${var.role}"
  }
  # Set a tag for use with codedeploy.  The Repo used for deployment should == role of these instances
  provisioner "local-exec" {
    command = "aws ec2 create-tags --resources ${self.id} --tags Key=instance-${var.owner}-${var.role}-${var.environment},Value=1 --profile=${var.account} --region=${var.region}"
  }
  lifecycle {
    ignore_changes = ["tags"]
  }
}

# DNS records for each host
resource "aws_route53_record" "instance_dns" {
  provider = "aws.twitch-aws" # We need to host DNS entries in twitch-aws
  count = "${var.count}"
  zone_id = "${var.r53_zone_id}"
  name = "${var.role}-${replace(element(aws_instance.instance.*.id, count.index), "i-", "")}.${lookup(var.environment_shortname, var.environment)}"
  type = "A"
  ttl = "300"
  records = ["${element(aws_instance.instance.*.private_ip, count.index)}"]
}
