# Use twitch-aws for DNS purposes
provider "aws" {
  alias   = "twitch-aws"
  region  = "us-west-2"
  profile = "twitch-aws"
}

provider "aws" {
  alias   = "${var.account}"
  region  = "${var.region}"
  profile = "${var.account}"
}

variable "template_name" {
  default = "user_data.tpl"
}

data "aws_iam_instance_profile" "aws_instance_profile" {
  name       = "${var.role}-${var.environment}"
  depends_on = ["aws_iam_instance_profile.instance_profile"]
}

# Basic IAM Role
resource "aws_iam_role" "iam_role" {
  provider = "aws.${var.account}"
  count    = "${var.instance_profile == "create" ? 1 : 0}"
  name     = "${var.role}-${var.environment}"
  path     = "/"

  assume_role_policy = <<EOF
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": "sts:AssumeRole",
      "Principal": {
        "Service": "ec2.amazonaws.com"
      },
      "Effect": "Allow",
      "Sid": ""
    }
  ]
}
EOF
}

resource "aws_iam_role_policy_attachment" "AmazonEC2RoleforSSM" {
  count      = "${var.instance_profile == "create" ? 1 : 0}"
  role       = "${aws_iam_role.iam_role.name}"
  policy_arn = "arn:aws:iam::aws:policy/service-role/AmazonEC2RoleforSSM"
}

# Instance profile for the instances
resource "aws_iam_instance_profile" "instance_profile" {
  provider   = "aws.${var.account}"
  name       = "${var.role}-${var.environment}"
  count      = "${var.instance_profile == "create" ? 1 : 0}"
  role       = "${aws_iam_role.iam_role.name}"
  depends_on = ["aws_iam_role.iam_role"]
}

# Load user_data
data "template_file" "user_data" {
  template = "${file(join("/", list(path.module, var.template_name)))}"

  vars {
    role        = "${var.role}"
    env         = "${lookup(var.environment_shortname, var.environment)}"
    owner       = "${var.owner}"
    puppet      = "${lookup(var.puppet_server, var.puppet)}"
    environment = "${var.environment}"
    dns_region  = "${lookup(var.dns_region, var.region)}"
    region      = "${var.region}"
  }
}

# AWS Instances
resource "aws_instance" "instance" {
  provider               = "aws.${var.account}"
  ami                    = "${var.ami}"
  count                  = "${var.count}"
  instance_type          = "${var.instance_type}"
  vpc_security_group_ids = ["${var.vpc_security_group_ids}"]

  # Stripe instances across the specified subnets
  subnet_id = "${element(var.subnet_ids, count.index % length(var.subnet_ids))}"

  root_block_device {
    volume_size = "${var.root_block_device_size}"
  }

  iam_instance_profile = "${var.instance_profile == "create" ? data.aws_iam_instance_profile.aws_instance_profile.name : var.instance_profile }"

  # Render and pass in the user_data template
  user_data = "${data.template_file.user_data.rendered}"

  tags {
    Name        = "${var.role}-${var.environment}"
    Environment = "${lookup(var.environment_tagname, var.environment)}"
    Owner       = "${var.owner}@twitch.tv"
    Service     = "${var.role}/${var.service}"
    Role        = "${var.role}"
    App         = "${var.service}"
    Team        = "${var.owner}"
  }

  depends_on = ["aws_iam_instance_profile.instance_profile"]

  # Set a tag for use with codedeploy.  The Repo used for deployment should == role of these instances
  provisioner "local-exec" {
    command = "aws ec2 create-tags --resources ${self.id} --tags Key=instance-${var.owner}-${var.role}-${var.environment},Value=1 Key=ServiceTuple:${var.chasqui_service}:${var.chasqui_stage},Value=:: --profile=${var.account} --region=${var.region}"
  }

  private_ip = "${lookup(var.private_ip, element(var.subnet_ids, count.index % length(var.subnet_ids)))}"
}

# DNS records for each host
resource "aws_route53_record" "instance_dns" {
  provider = "aws.twitch-aws"                                                                                                                           # We need to host DNS entries in twitch-aws
  count    = "${var.count}"
  zone_id  = "${lookup(var.r53_zone_id, var.region)}"
  name     = "${var.role}-${replace(element(aws_instance.instance.*.id, count.index), "i-", "")}.${lookup(var.environment_shortname, var.environment)}"
  type     = "A"
  ttl      = "300"
  records  = ["${element(aws_instance.instance.*.private_ip, count.index)}"]
}
